/*********************************************************************
 *
 *                HP-Print
 *
 *********************************************************************
 * FileName:        mainwindow.cpp
 * Compiler etc.    GCC, Qt5.11.3
 * Author:          sprut (sprut@sprut.de)
 *
 * Software License Agreement
 *
 *  HP-Print,   convert outputs from HP8595 into a PNG-Graphic
 *   Copyright (C) 2021  Bredendiek (sprut)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <https://www.gnu.org/licenses/>.
 ***********************************************************************/

#include <iomanip>
#include <sstream>
#include <time.h>
#include <sys/time.h>
#include <cstdio>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>  // write(), read(), close()

#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "settingsdialog.h"

#include <QMessageBox>
#include <QFile>
#include <QFileDialog>
#include <QColor>


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),    
    ui(new Ui::MainWindow),
    m_settings(new SettingsDialog),
    m_serial(new QSerialPort(this))
{
    //Timer to initiate picture generation
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(TimerAction()));
    timer->start(interval_ms);

    paint_enable = false;

    ui->setupUi(this);
    showStatusMessage(tr("Port Closed"));

    RxBuffer.clear();

    //hide if not debug
    ui->label->setText("Bytes received: 0");
    ui->lbl8->setText("Printer:");
    ui->lbl9->setText("");
    ui->lbl10->setText("");

    ui->actionConnect->setEnabled(true);
    ui->actionDisconnect->setEnabled(false);
    ui->actionEnd->setEnabled(true);
    ui->actionSelect_Port->setEnabled(true);
    ui->pushButton_2->setVisible(debug);
    ui->pushButton_3->setVisible(debug);

    initActionsConnections();
    connect(m_serial, &QSerialPort::readyRead, this, &MainWindow::readData);
    connect(m_serial, &QSerialPort::errorOccurred, this, &MainWindow::handleError);

    buffer_pointer = 0;
} // MainWindow


void MainWindow::TimerAction(){
  if (!rx_active) rx_finished=true;
  rx_active = false;

  // daten liegen in RxBuffer
  if (rx_finished) {
    if (RxBuffer.size()>10000) {
        ui->label->setText("Bytes : "+QString::number( RxBuffer.size() ));
        paint_enable = true;
        toPNG = false;
        update();
    }
  }
} //TimerAction


// read unsigned from RxBuffer
int MainWindow::getRxBufferInt(int p)
{
    int n = 0;
    if (p<RxBuffer.size()) {
        n = RxBuffer.at(p);
        if (n<0) n = n+256;
    }
    return n;
}


void MainWindow::paintEvent(QPaintEvent *event)
{
    if(!paint_enable) return;
    if (RxBuffer.size()<10000) return;

    showStatusMessage(tr("Printing"));

    QImage img(this->size(), QImage::Format_ARGB32);
    img.fill(Qt::white);

    //Painter vorbereiten
    QPainter painter;
    if (toPNG) {
        painter.begin(&img);
    } else {
        painter.begin(this);
    }

    QPen myPen(Qt::black, 2, Qt::SolidLine);
    myPen.setColor(Qt::red);

    char stretch=1;
    int x1 = 20;
    int y1 = 100;
    int x2 = x1;
    int y2 = y1;
    int xx = x1;
    int yy = y1;
    int drucker = 0;
    char klein = ' ';
    char gross = ' ';
    int breit  = 512;
    int wert;
    char zeichen;
    int col_R=0, col_G=0, col_B=0;
    int plane = 0;
    int raster[4][200];
    for (int i=0; i<200; i++) {raster[0][i] = 0; raster[1][i] = 0; raster[2][i] = 0; raster[3][i] = 0;}

    int p = 0;
    int line_spacing;
    int n1;
    int n2;
    int lang;
    QColor palette[32];
    // default is CYK
    for (int k=0;k<31;k++) palette[k] = Qt::black;
    palette[0] = Qt::white;
    palette[1] = Qt::cyan;
    palette[2] = Qt::magenta;
    palette[3] = Qt::blue;
    palette[4] = Qt::yellow;
    palette[5] = Qt::green;
    palette[6] = Qt::red;
    palette[7] = Qt::black;

    while (p<RxBuffer.size()) {
        //ui->lbl12->setText(QString::number( p ));
        if (RxBuffer.at(p) == 0x1B) {
            p++;
            switch (RxBuffer.at(p))
            {

            // HP color/bw
            case 0x2A:
            case 0x26:
                p++;
                //pointer zeigt auf Kleinbuchstaben
                klein = RxBuffer.at(p);
                wert = 0;
                zeichen = 0;
                p++;
                if (getRxBufferInt(p) > 0x39) {
                    zeichen = RxBuffer.at(p);
                    p++;
                }
                while (getRxBufferInt(p) <0x3A) {
                    wert = wert*10 + (getRxBufferInt(p) - 0x30);   // *rbC = 1B-2A-72-62-43
                    p++;
                }
                gross = RxBuffer.at(p);
                switch (klein)
                {
                case 't':     // set raster resolution          *t#R
                    break;
                case 'a':     // horiz cursor position          &a#H
                    //xx = wert;
                    break;
                case 'b':
                    switch (gross){
                    case 'M':   // set compression mode         *b#M
                        break;
                    case 'V':   // send raster data by plane    *b#V
                        if ((wert==64)&&(drucker==0))  drucker = 3; // hp_deskjet_color
                        if (wert==192) drucker = 4;                 // hp_deskjet_expand
                        for (int i=0; i<wert; i++) {
                            raster[plane][i] = RxBuffer.at(++p);
                            raster[3][i] = 0;
                        }
                        plane++;
                        break;
                    case 'W':   // send raster data             *b#W
                        for (int i=0; i<wert; i++) raster[plane][i] = RxBuffer.at(++p);
                        for (int i=0; i<wert; i++) {
                            for (int m=128; m>0; m= m /2) {
                                int farbe;
                                farbe = 0;
                                if (raster[0][i] & m) farbe = farbe +1;
                                if (raster[1][i] & m) farbe = farbe +2;
                                if (raster[2][i] & m) farbe = farbe +4;
                                if (raster[3][i] & m) farbe = farbe +8;
                                myPen.setColor(palette[farbe]);
                                painter.setPen(myPen);
                                painter.drawPoint(xx,yy);
                                if (stretch) painter.drawPoint(xx,yy+1);
                                xx++;
                                if (x2<xx) x2 = xx;
                            }
                        }
                        xx = x1;
                        yy++;
                        if (stretch) yy++;
                        if (y2<yy) y2 = yy;
                        plane = 0;
                        break;
                    }
                    break;
                case 'r':
                    switch (gross){
                    case 'S':    // set raster area width       *r#S
                        breit = wert;
                        break;
                    case 'U':    //  set simple color           *r-3U = CYK    *r4U = 16farbenpalette   *r1U = bw
                        if (wert==1) {
                            palette[0] = Qt::white;
                            palette[1] = Qt::black;
                            drucker = 6;    // HP-BW
                        }
                        break;
                    case 'A':    // start raster transfer       *r#A
                        break;
                    case 'C':    // end raster                  *r#C
                        p = RxBuffer.size();
                        break;
                    }
                    break;
                case 'v':
                    switch (gross){
                    case 'A':    // set red  *v#A
                        col_R = wert;
                        break;
                    case 'B':    //  set green
                        col_G = wert;
                        break;
                    case 'C':    // start blue
                        col_B = wert;
                        break;
                    case 'I':    // start color to palette
                        drucker = 5;  // paintjet
                        palette[wert] = QColor(col_R,col_G,col_B);
                        col_R = 0;
                        col_G = 0;
                        col_B = 0;
                        break;
                    }
                    break;
                 }
                break; // case 26

            //Epson
            case 60:  // unidirektional mode for line
                break;
            case 79:  // cancel  skip over perforation
                break;
            case 51:  //Select n/216 inch line spacing (n=0..255)
                p++;
                line_spacing = getRxBufferInt(p);
                break;
      //      case 42:     //1B 2A m n 1 n 2 d 1 .. d x     das überschneidet sich mit HP-code

      //          break
            case 75:                        // 4B   60 dpi ep_mx80_lrg
                drucker = 1;
                p++;
                n1 = getRxBufferInt(p);
                p++;
                n2 = getRxBufferInt(p);
                lang = n1 + (n2<<8);
                for (int s=0; s<lang;s++) {
                    p++;
                    if (RxBuffer.at(p) & 0x80) painter.drawPoint(xx,yy);
                    if (RxBuffer.at(p) & 0x40) painter.drawPoint(xx,yy+1);
                    if (RxBuffer.at(p) & 0x20) painter.drawPoint(xx,yy+2);
                    if (RxBuffer.at(p) & 0x10) painter.drawPoint(xx,yy+3);
                    if (RxBuffer.at(p) & 0x08) painter.drawPoint(xx,yy+4);
                    if (RxBuffer.at(p) & 0x04) painter.drawPoint(xx,yy+5);
                    if (RxBuffer.at(p) & 0x02) painter.drawPoint(xx,yy+6);
                    if (RxBuffer.at(p) & 0x01) painter.drawPoint(xx,yy+7);
                    xx++;
                    if (x2<xx) x2 = xx;
                }
                yy = yy + 8;
                xx = x1;
                if (y2<yy) y2 = yy;
                break;
            case 76:                            // 4C  120 dpi ep_mx80_sml
                drucker = 2;
                p++;
                n1 = getRxBufferInt(p);
                p++;
                n2 = getRxBufferInt(p);
                lang = n1 + (n2<<8);
                for (int s=0; s<lang;s++) {
                    p++;
                    if (RxBuffer.at(p) & 0x80) {painter.drawPoint(xx,yy);    painter.drawPoint(xx,yy+1); }
                    if (RxBuffer.at(p) & 0x40) {painter.drawPoint(xx,yy+2);  painter.drawPoint(xx,yy+3); }
                    if (RxBuffer.at(p) & 0x20) {painter.drawPoint(xx,yy+4);  painter.drawPoint(xx,yy+5); }
                    if (RxBuffer.at(p) & 0x10) {painter.drawPoint(xx,yy+6);  painter.drawPoint(xx,yy+7); }
                    if (RxBuffer.at(p) & 0x08) {painter.drawPoint(xx,yy+8);  painter.drawPoint(xx,yy+9); }
                    if (RxBuffer.at(p) & 0x04) {painter.drawPoint(xx,yy+10); painter.drawPoint(xx,yy+11);}
                    if (RxBuffer.at(p) & 0x02) {painter.drawPoint(xx,yy+12); painter.drawPoint(xx,yy+13);}
                    if (RxBuffer.at(p) & 0x01) {painter.drawPoint(xx,yy+14); painter.drawPoint(xx,yy+15);}
                    xx++;
                    if (x2<xx) x2 = xx;
                }
                yy = yy + 16;
                xx = x1;
                if (y2<yy) y2 = yy;
                break;
            default:  ;
            }
            p++;
        }
        else {
            p++; // eigentlich haette hier esc stehen sollen
        }
    }
    painter.end();

    switch (drucker)
    {
        case 1: ui->lbl8->setText("ep_mx80_lrg");       break;
        case 2: ui->lbl8->setText("ep_mx80_sml");       break;
        case 3: ui->lbl8->setText("hp_deskjet_color");  break;
        case 4: ui->lbl8->setText("hp_deskjet_expand"); break;
        case 5: ui->lbl8->setText("hp_paintjet");       break;
        case 6: ui->lbl8->setText("hp_bw_printer");     break;
        default: ui->lbl8->setText("?");
    }
    x2--;
    y2--;
//    RxBuffer.clear();
    paint_enable = false;
    ui->lbl9->setText("x1 .. x2: "+QString::number( x1 )+" ... "+QString::number( x2 ));
    ui->lbl10->setText("y1 .. y2: "+QString::number( y1 )+" ... "+QString::number( y2 ));

    if (toPNG) {
        const char *homedir;
        if ((homedir = getenv("HOME")) == NULL) {   // existiert env-variable HOME?
            homedir = getpwuid(getuid())->pw_dir;   // wenn nicht, dann von getuid
        }
        char filename[255];
        struct tm* tm;
        time_t now;
        now = time(0);                          // get current time
        tm = localtime(&now);                   // get structure
        sprintf(filename, "%s/HP8595_%04d_%02d_%02d-%02d:%02d:%02d.PNG", homedir, tm->tm_year+1900, tm->tm_mday, tm->tm_mon+1, tm->tm_hour, tm->tm_min, tm->tm_sec  );
        img.save(filename);
        showStatusMessage(tr("Saving PNG"));
    }
} //PaintEvent


// is called from serial port if data is arrived
// reads data from the serial port
// fills a set of data into RxBuffer
void MainWindow::readData() {
    const QByteArray data = m_serial->readAll();
    if (rx_finished) RxBuffer.clear();
    rx_active   = true;
    rx_finished = false;
    // an den puffer anhängen
    RxBuffer.append(data);
    showStatusMessage(tr("Receiving"));
    ui->label->setText("Bytes received: "+QString::number( RxBuffer.size() ));
} //readData


// called from serial port in case of error
void MainWindow::handleError(QSerialPort::SerialPortError error)
{
    if (error == QSerialPort::ResourceError) {
        QMessageBox::critical(this, tr("Critical Error"), m_serial->errorString());
        closeSerialPort();
    }
}


// closes the serial port
void MainWindow::closeSerialPort()
{
    if (m_serial->isOpen()) m_serial->close();
    ui->actionConnect->setEnabled(true);
    ui->actionDisconnect->setEnabled(false);
    ui->actionSelect_Port->setEnabled(true);
    ui->pushButton->setEnabled(true);
    showStatusMessage(tr("Disconnected"));
}


void MainWindow::initActionsConnections()
{
    connect(ui->actionConnect, &QAction::triggered, this, &MainWindow::OpenPort);
    connect(ui->actionDisconnect, &QAction::triggered, this, &MainWindow::closeSerialPort);
    connect(ui->actionEnd, &QAction::triggered, this, &MainWindow::close);
    connect(ui->actionSelect_Port, &QAction::triggered, m_settings, &SettingsDialog::fillAndShow);//show);
//    connect(ui->actionAbout, &QAction::triggered, this, &MainWindow::about);
    connect(ui->actionAbout_QT, &QAction::triggered, qApp, &QApplication::aboutQt);
}


// destructor
MainWindow::~MainWindow()
{
    delete ui;
    delete m_settings;
}


// call OpenPort
void MainWindow::on_pushButton_clicked()
{
    OpenPort();
}


// opens the selected port with parameters 2400-7-1-o
void MainWindow::OpenPort() {
    if (connected) return;

    RxBuffer.clear();
    const SettingsDialog::Settings p = m_settings->settings();
    m_serial->setPortName(p.name);
    m_serial->setBaudRate(p.baudRate);
    m_serial->setDataBits(p.dataBits);
    m_serial->setParity(p.parity);
    m_serial->setStopBits(p.stopBits);
    m_serial->setFlowControl(p.flowControl);
    if (m_serial->open(QIODevice::ReadWrite)) {
        ui->actionConnect->setEnabled(false);
  //      ui->actionDisconnect->setEnabled(true);
        ui->actionSelect_Port->setEnabled(false);
        ui->pushButton->setEnabled(false);
        connected = 1;
        showStatusMessage(tr("Connected to %1 : %2, %3, %4, %5, %6")
                          .arg(p.name).arg(p.stringBaudRate).arg(p.stringDataBits)
                          .arg(p.stringParity).arg(p.stringStopBits).arg(p.stringFlowControl));
    } else {
        QMessageBox::critical(this, tr("Error"), m_serial->errorString());
        showStatusMessage(tr("Open error"));
        connected = 0;
    }
} //OpenPort


// put a string into the statusbar
void MainWindow::showStatusMessage(const QString &message)
{    
    statusBar()->showMessage(message);
}


// programm info
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::information(this, "HP-Print", "Receives the Print-Data-Stream from a HP8595.\n "
                                               "and converts it into a Graphic.\n ");
}



// from file
void MainWindow::on_pushButton_3_clicked()
{
    // Get the file name using a QFileDialog
    QFile file(QFileDialog::getOpenFileName(NULL, tr("Select Print-File")));

    // If the selected file is valid, continue with the upload
    if (!file.fileName().isEmpty()) {
        file.open(QIODevice::ReadOnly);
        RxBuffer.clear();
        RxBuffer = file.readAll();
        file.close();
        if (RxBuffer.at(0) == 0x1B) {
            rx_finished = true;
            //darstellung macht der Timer
        } else {
            RxBuffer.clear();
            showStatusMessage(tr("File does not look good to me."));
        }
    }
}


void MainWindow::SaveToFile(){
    const char *homedir;
    if ((homedir = getenv("HOME")) == NULL) {   // existiert env-variable HOME?
        homedir = getpwuid(getuid())->pw_dir;   // wenn nicht, dann von getuid
    }
    char filename[255];
    struct tm* tm;
    time_t now;
    now = time(0);                          // get current time
    tm = localtime(&now);                   // get structure
    sprintf(filename, "%s/printfile_%04d_%02d_%02d-%02d:%02d:%02d.prn", homedir, tm->tm_year+1900, tm->tm_mday, tm->tm_mon+1, tm->tm_hour, tm->tm_min, tm->tm_sec  );
    QFile newDoc(filename);
    if(newDoc.open(QIODevice::WriteOnly)){
        newDoc.write(RxBuffer);
    }
    newDoc.close();
    showStatusMessage(tr("Data saved to file "));
}


void MainWindow::on_actionClear_triggered()
{
    RxBuffer.clear();
    update();
}


// Buf -> File
void MainWindow::on_pushButton_2_clicked()
{
   SaveToFile();
}


//to PNG
void MainWindow::MakePicture()
{
    paint_enable = true;
    toPNG = true;
    update();
//    toPNG = false;
}


void MainWindow::on_pushButton_5_clicked()
{
   MakePicture();
}
