/*********************************************************************
 * FileName:        user.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Company:         sprut
 * Copyright:       2007-2012 Joerg Bredendiek (sprut)
 *
 *
 ********************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#define	NO_CDC	//MCD
//#define	CDC_HEX	// nonsens
//#define	CDC_ASCII


/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include <usart.h>
#include "system\typedefs.h"
#include "system\usb\usb.h"
#include "io_cfg.h"             // I/O pin mapping

#include "delays.h"
//#include "system\interrupt\interrupt.h"
#include "user\user.h"
#include "timers.h"
#include "i2c.h"
#include "pwm.h"
#include "mwire.h"
#include "spi.h"


/** V A R I A B L E S ********************************************************/
#pragma udata

#ifdef	CDC_ASCII
char	input_buffer[64];
byte	rgetascii;
byte 	hb_counter 		= 0;
byte	ib_counter 		= 0;
char	asciiKor   		= 0;
byte	befehl     		= 0;	//eine gueltige Befehlszeile wurde empfangen
byte	wertgueltig  	= 0; //steht in rgetascii schon was?
byte	zeileangefangen = 0; //kam schon mal ein #?
#endif	//CDC_ASCII


byte				counter;
DATA_PACKET 		dataPacket;


// LCD
byte				LCD_zeilen  = 2;
byte				LCD_spalten = 16;
byte				LCD_1x16	= 0;	

// IO-Pins
byte				MaskeA		= 0x7F;  // welche Portbits duerfen manipuliert werden? RA7 fehlt
byte				MaskeB		= 0xFF;
byte				MaskeC		= 0xC7;	// RC4&RC5=USB, RC3 fehlt

//ADC
byte				AcdMaskeA	= 0;	// von ADC verwendete Pins
byte				AcdMaskeB	= 0;	// von ADC verwendete Pins

// SPI
byte				SPI_SlaveMode = 0;
byte				SPI_Maske	= 0; 	// im PortB belegte Pins fuer Select-Leitungen der SPI-Slaves

//Schieberegister
byte				SR_Mode		= 0;
byte				SR_Mode_r	= 0;
byte				SR_Delay	= 100;

//schrittmotor
T_STEP				ste[5];				// ABCD Schrittmotor

//L297 Schrittmotor
T_L297				L297[5];

//RS-232
byte				RS232_running = 0;
#ifdef	CDC_ASCII
byte				RS232_buffer[21];	// Empfangspuffer
#else
byte				RS232_buffer[32];	// Empfangspuffer
#endif
byte				RS232_pointer = 0;	// zeigt auf naechste freie stelle in RS232_buffer

//Servos
byte				ServoB_running = 0;
BYTE				ServoBMaske	   = 0;
byte				ServoBPuls[8]  = {50,50,50,50,50,50,50,50};
byte				ServoC_running = 0;
BYTE				ServoCMaske	   = 0;
byte				ServoCPuls[8]  = {50,50,50,50,50,50,50,50};
word				ServoNull      = 1000;

/** D E C L A R A T I O N S **************************************************/
#pragma code
//Aufruf nach Reset
void UserInit(void)
{
	char k;

	// init LCD
	LCD_1x16 = 0;

	// init pins
	MaskeA	= 0x7F;  // welche Portbits duerfen manipuliert werden?
	MaskeB	= 0xFF;
	MaskeC	= 0xC7;
	TRISA 	= 0xFF;
	TRISB 	= 0xFF;
	TRISC 	= 0xFF;

	// init ABCD-Schrittmotor
	for (k=0; k<5; k++)
	{
		ste[k].step		= 0;
		ste[k].running	= 0;
		ste[k].mode 	= 0;
		ste[k].periode	= 1;
		ste[k].schritte = 0;
		ste[k].gone		= 0;
		StepBeschInit(k);		// Beschleunigungstabelle belegen
	}

	// init L297-Schrittmotor
	for (k=0; k<5; k++)
	{
		L297[k].running		= 0;
		L297[k].mode._byte	= 0;
		L297[k].periode 	= 1;
		L297[k].schritte	= 1;
	}

}//end UserInit




/******************************************************************************
 * Function:        void ProcessIO(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is a place holder for other user routines.
 *                  It is a mixture of both USB and non-USB tasks.
 *
 * Note:            None
 *****************************************************************************/
void ProcessIO(void)
{   
    // User Application USB tasks
    if((usb_device_state < CONFIGURED_STATE)||(UCONbits.SUSPND==1)) return;
    ServiceRequests();
}//end ProcessIO



// A S C I I *****************************************************************************
#ifdef CDC_ASCII

// ermittelt, ob es sich bei einem Zeichen um ein Trennzeichen handelt
// ermittle den noetigen Wert fuer asciiKor
// alles ausser 0..9/a..f/A.F sind trennzeichen
// aus A..F/a..F wird 10..15 (0x0A..0x0F)
// input: zeichen
// output:	-1 : Trennzeichen
//			-2 : Zeilenende
//			-3 : Zeilenanfang
//			 0 : ASCII-Zeichen
byte	trennzeichen(char	zeichen)
{
	if (zeichen== '#') return -3; //SOL
	if (zeichen==0x00) return -2; //EOL
	if (zeichen==0x0A) return -2; //EOL
	if (zeichen==0x0D) return -2; //EOL

	asciiKor = 0x00;
	if (zeichen < '0') return -1;	//   .. /
	if (zeichen <='9') 				// 0 .. 9
	{
		asciiKor = '0';
		return 0;
	}
	if (zeichen < 'A') return -1;	// : .. @
	if (zeichen <='F') 				// A .. F
	{
		asciiKor = 'A'-10;
		return 0;
	}
	if (zeichen < 'a') return -1;	// G .. '
	if (zeichen <='f') 				// a .. f
	{
		asciiKor = 'a'-10;
		return 0;
	}
	return -1;						// g .. 
} //trennzeichen


// input_buffer[ib_counter] steht auf letztem Trennzeichen
// input_buffer lesen bis zum nchsten Trennzeichen
// wandeln in Hex -> rgetascii
// bei Zeilenende  -1
byte	getascii(void)
{
	rgetascii = 0;
	while (trennzeichen(input_buffer[ib_counter]) == -1) ib_counter++;  // trennzeichen bersprinegn
	if    (trennzeichen(input_buffer[ib_counter]) == -2) return (-1);	//EOL
//	if (ib_counter >= mCDCGetRxLength())               return (-1);	    //EOL
	while (trennzeichen(input_buffer[ib_counter]) ==  0)
	{
		rgetascii <<= 4;
		rgetascii += (input_buffer[ib_counter] - asciiKor);	
		ib_counter++;
	}
	return(0);
}	//getascii


//input_buffer -> dataPacket
//  #xx-xx-xx-xx-xx-xx
// einlesen der per RS232 empfangenen Zeichen
// beim Zeilenende wird befehl=1 gesetzt
void	AsciiToHexParser(void)
{
	for (ib_counter = 0; ib_counter < mCDCGetRxLength(); ib_counter++)
	{
		switch (trennzeichen(input_buffer[ib_counter]))
		{
			case -3:									//zeilenanfang
				for (hb_counter = 0; hb_counter < 64; hb_counter++)
					dataPacket._byte[hb_counter] = 0;
				hb_counter  = 0;
				befehl      = 0;
				rgetascii   = 0;
				wertgueltig = 0;
				zeileangefangen = 1;
				break;
			case -2:									//zeilenende
				if (wertgueltig == 1)
				{
					dataPacket._byte[hb_counter] = rgetascii;
					hb_counter++;
					wertgueltig = 0;
				}
				rgetascii = 0;
				if (zeileangefangen == 1) befehl = 1;
				break;
			case -1:									//trennzeichen
				if (wertgueltig == 1)
				{
					dataPacket._byte[hb_counter] = rgetascii;
					hb_counter++;
					wertgueltig = 0;
				}
				rgetascii = 0;
				break;
			case 0: 									//ziffer
				rgetascii <<= 4;
				rgetascii += (input_buffer[ib_counter] - asciiKor);	
				wertgueltig = 1;						
				break;
		} //switch
	}
}	//AsciiToHexParser


// wandelt Zahl (0..15) in Zeichen ('0' .. 'A')
char	zeichen( byte wert)
{
	if (wert < 10) return (wert+'0');
	return (wert-10+'A');
}


//dataPacket -> input_buffer
void	HexToAsciiParser(void)
{
	char	puffer;
	for (ib_counter= 0; ib_counter<64; ib_counter++)
		input_buffer[ib_counter] = 0;
	ib_counter = 1;
	hb_counter = 0;
	input_buffer[0] = '@';
	for (hb_counter= 0; hb_counter<16; hb_counter++)
	{
		puffer = dataPacket._byte[hb_counter];
		input_buffer[ib_counter] = zeichen((puffer & 0xF0) >> 4);
		ib_counter++;
		input_buffer[ib_counter] = zeichen( puffer & 0x0F);
		ib_counter++;
		input_buffer[ib_counter] = '-';
		ib_counter++;
	}
	input_buffer[ib_counter] = 0x0A;
	ib_counter++;
	input_buffer[ib_counter] = 0x0D;
	ib_counter++;
	input_buffer[ib_counter] = 0x00;
	counter = ib_counter;
}

#endif //CDC_ASCII



// Kommando auswerten und ausfhren **********************
void ServiceRequests(void)
{
    byte	index;
	word	big_counter;
	WORD	uidaten;
  
    // wurde via USB etwas empfangen?

	#ifdef NO_CDC	// MCD Treiber
    if(USBGenRead((byte*)&dataPacket,sizeof(dataPacket)))  //byte USBGenRead(byte *buffer, byte len)	//normal
	#endif	//NO_CDC

	#ifdef CDC_HEX	//CDC-Treiber im HEX-Format (nonsens)
    if(getsUSBUSART((char*)&dataPacket,sizeof(dataPacket)))  //byte getsUSBUSART(char *buffer, byte len)	//cdc hex
	#endif	//CDC_HEX

	#ifdef CDC_ASCII	// CDC Treiber
    if(getsUSBUSART((char*)&input_buffer,sizeof(input_buffer)))  //byte getsUSBUSART(char *buffer, byte len)	//cdc ascii
		AsciiToHexParser();
	if (befehl==1)
	#endif	//CDC_ASCII

    {
        counter = 0;

		#ifdef CDC_ASCII
		befehl = 0;
		#endif	//CDC_ASCII

		switch(dataPacket.CMD) 		//das ist in Assembler ein riesieger Sprungverteiler 
        {
			// eine Kennummer aus dem PIC auslesen
            case READ_VERSION:
                //dataPacket._byte[1] is len
                dataPacket._byte[2] = MINOR_VERSION; // Firmware-Version steht in user.h
                dataPacket._byte[3] = MAJOR_VERSION; // Hardware-Version
                counter=0x04;
                break;

			//**********************************************************************//
			// IO, ADC, FRQin, I2C, SPI, LCD, TMR1-3, PWM1, PWM2, RS232, EEPROM, KEY, LED

			case SYS_IO:
				UP_IO();
   				break;

			case SYS_ADC:
				UP_ADC();
   				break;

			case SYS_FRQ:
				UP_FRQ();
   				break;

			case SYS_RS232:
				UP_RS232();
   				break;

			case SYS_I2C:
				UP_I2C();
   				break;

			case SYS_SPI:
				UP_SPI();
   				break;

			case SYS_MWIRE:
				UP_MWIRE();
   				break;

			case SYS_SR:
				UP_SR();
   				break;

			case SYS_LCD1:
				UP_LCD();
   				break;

			case SYS_LCD2:
				UP_LCD();
   				break;

			case SYS_PWM1:
				UP_PWM1();
   				break;

			case SYS_PWM2:
				UP_PWM2();
   				break;

			case SYS_EEPROM:
				UP_EEPROM();
   				break;

			case SYS_STEP1:
				UP_STEPuni(1);
   				break;

			case SYS_STEP2:
				UP_STEPuni(2);
   				break;

			case SYS_STEP3:
				UP_STEPuni(3);
   				break;

			case SYS_STEP4:
				UP_STEPuni(4);
   				break;

			case SYS_L297_1:
				UP_L297_Uni(1);
				//UP_L297_1();
   				break;
                
			case SYS_L297_2:
				UP_L297_Uni(2);
				//UP_L297_2();
   				break;
                
			case SYS_L297_3:
				UP_L297_Uni(3);
				//UP_L297_3();
   				break;
                
			case SYS_L297_4:
				UP_L297_Uni(4);
				//UP_L297_4();
   				break;
               
			case SYS_ServoB:
				UP_ServoB();
   				break;
                
			case SYS_ServoC:
				UP_ServoC();
   				break;
                 
			case SYS_COUNTER_0:
				UP_Counter0();
   				break;
                 
			case SYS_COUNTER_3:
				UP_Counter3();
   				break;
                 
            case RESET:
                //When resetting, make sure to drop the device off the bus
                //for a period of time. Helps when the device is suspended.
                UCONbits.USBEN = 0;
                big_counter = 0;		//word = 16 Bit
                while(--big_counter);
				SleepMs(250);
				SleepMs(250);

                Reset();
                break;
                
            default:
				dataPacket._byte[0] = 0xFF;		// Error, unbekannter Befehl
				counter = 16;
                break;
        }//end switch()


		#ifdef NO_CDC		//senden mit microchip custom driver
		if(counter != 0)
		{
			if(!mUSBGenTxIsBusy())
				USBGenWrite((byte*)&dataPacket,counter);
		}
		#endif


		#ifdef CDC_ASCII		//senden mit CDC-Treiber im ASCII-code
		if(counter != 0)
		{
			HexToAsciiParser();	//wandlung in ASCII-String
			if(mUSBUSARTIsTxTrfReady())
				mUSBUSARTTxRam((byte*)&input_buffer,counter);
		}
		#endif	//CDC_ASCII


		#ifdef CDC_HEX		//senden mit CDC-Treiber im HEX-Format (nonsens)
		if(counter != 0)
		{
			if(mUSBUSARTIsTxTrfReady())
				mUSBUSARTTxRam((byte*)&dataPacket,counter);
		}
		#endif	//CDC_HEX


    }	//end if wurde via USB entwas empfangen 


	// folgende Routinen werden bei jedem Schleifendurchlauf durchlaufen
	if (RS232_running == 1) RS232_run();

	if (ste[1].running == 1) stepUnirun(1);
	if (ste[2].running == 1) stepUnirun(2);
	if (ste[3].running == 1) stepUnirun(3);
	if (ste[4].running == 1) stepUnirun(4);

	if (L297[1].running) L297_Unirun(1);
	if (L297[2].running) L297_Unirun(2);
	if (L297[3].running) L297_Unirun(3);
	if (L297[4].running) L297_Unirun(4);

	if (ServoB_running == 1) ServoB_run();
	if (ServoC_running == 1) ServoC_run();
}//end ServiceRequests


/*************

0 - System/Interface
1 - 0=aus/1=init/2=write/3=read/...
2 - daten
3 - daten
4 - daten
5 - daten
6 - daten
7 - daten

**************/


//*********************************************************************************************//

//** I O - P I N S ****************************************************************************//

// IO-Ports A..C
// 0 - aus:   alle Pin auf input
// 1 - init:  TRIS ganzer ports und pull-up
// 2 - write: schreiben ganzer ports
// 3 - read:  lesen ganzer ports

// 4 - setzen einzelner TRIS-Bits (die mit 1)
// 5 - loeschen einzelner TRIS-Bits (die mit 1)
// 6 - setzen einzelner Pins (die mit 1)
// 7 - loeschen einzelner pins (die mit 1)
void UP_IO(void)
{
	switch (dataPacket._byte[1])
	{
		case 0://aus:   alle Pin auf input
			TRISA |= MaskeA;
			TRISB |= MaskeB;
			TRISC |= MaskeC;
			break;
		case 1://init:  TRIS ganzer ports und pull-up
			TRISA = dataPacket._byte[2] & MaskeA;
			TRISB = dataPacket._byte[3] & MaskeB;
			TRISC = dataPacket._byte[4] & MaskeC;
			if (dataPacket._byte[5] & 1 == 1)	INTCON2bits.NOT_RBPU = 0; //pullup an
				else 							INTCON2bits.NOT_RBPU = 1; //pullup aus
			break;
		case 2://write: schreiben ganzer ports
			LATA = (LATA & (~MaskeA)) | (dataPacket._byte[2] & MaskeA);
			LATB = (LATB & (~MaskeB)) | (dataPacket._byte[3] & MaskeB);
			LATC = (LATC & (~MaskeC)) | (dataPacket._byte[4] & MaskeC);
			break;
		case 3://read:  lesen ganzer ports
			dataPacket._byte[2] = PORTA;
			dataPacket._byte[3] = PORTB;
			dataPacket._byte[4] = PORTC;
			break;
		case 4://setzen einzelner TRIS-Bits (die mit 1)
			TRISA |= (dataPacket._byte[2] & MaskeA);
			TRISB |= (dataPacket._byte[3] & MaskeB);
			TRISC |= (dataPacket._byte[4] & MaskeC);
			break;
		case 5://loeschen einzelner TRIS-Bits (die mit 1)
			TRISA &= ~(dataPacket._byte[2] & MaskeA);
			TRISB &= ~(dataPacket._byte[3] & MaskeB);
			TRISC &= ~(dataPacket._byte[4] & MaskeC);
			break;
		case 6://setzen einzelner Pins (die mit 1)
			LATA |= (dataPacket._byte[2] & MaskeA);
			LATB |= (dataPacket._byte[3] & MaskeB);
			LATC |= (dataPacket._byte[4] & MaskeC);
			break;
		case 7://loeschen einzelner pins (die mit 1)
			LATA &= ~(dataPacket._byte[2] & MaskeA);
			LATB &= ~(dataPacket._byte[3] & MaskeB);
			LATC &= ~(dataPacket._byte[4] & MaskeC);
			break;
	}
	counter = 16;
} // UP_IO



//** A D C *******************************************************************************//

// ADC mit AN0..4, 10 Bit, Ref=Vdd/Vss
// AN0..4 = RA0,1,2,3,5
// AN8..12= RB2,3,1,4,0  was fuer ein Chaos!
// Vref+ = AN3
// Vref- = AN2
//
// 0 - ADC aus
// 1 - Zahl der Eingaenge 0..5, einschalten, Vref festlegen
// 2 - aktuellen Eingang  0..4 festlegen
// 3 - Resultat auslesen
void UP_ADC(void)
{
	byte	channel;
	switch (dataPacket._byte[1])
	{
		case 0://ADC aus
			ADCON0    = 0x00; 		// AN1, ADC-aus
			ADCON1    = 0x0F; 		// alles digital
			MaskeA   |= AcdMaskeA;	// Freigabe der verwendeten ADC-Pins
			MaskeB   |= AcdMaskeB;	// Freigabe der verwendeten ADC-Pins
			AcdMaskeA = 0x00;
			AcdMaskeB = 0x00;
			break;
		case 1: //ADC an, Zahl der Eingaenge 0..5 festlegen
			ADCON0 = 0x00; 		             // ADC-aus
			ADCON2 = 0xAA;		             // right-justified, Aquisi=12Tad, Fad=Fosc/32 (schnell)
			switch (dataPacket._byte[2])
			{
				case 0:	//alles digital
					ADCON1     = 0x0F; 
					MaskeA = (MaskeA & 0xD0) | 0x2F;
					MaskeB = (MaskeB & 0xE0) | 0x1F;
					break;
				case 1:	// AN0
					TRISA     |= 0x01; 
					AcdMaskeA |= 0x01;
					MaskeA = (MaskeA & 0xD0) | 0x2E;
					ADCON1 = 0x0E;
					break;
				case 2:	// AN0..1
					TRISA     |= 0x03;
					AcdMaskeA |= 0x03;
					MaskeA = (MaskeA & 0xD0) | 0x2C;
 					ADCON1 = 0x0D;
					break;
				case 3:	// AN0..2
					TRISA     |= 0x07;
					AcdMaskeA |= 0x07;
					MaskeA = (MaskeA & 0xD0) | 0x28;
  					ADCON1 = 0x0C;
					break;
				case 4:	// AN0..3
					TRISA     |= 0x0F;
					AcdMaskeA |= 0x0F;
					MaskeA = (MaskeA & 0xD0) | 0x20;
   					ADCON1 = 0x0B;
					break;
				case 5:	// AN0..4
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
   					ADCON1 = 0x0A;
					break;
				case 6:	// AN0..4 AN8
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
					TRISB     |= 0x04;	//RB2
					AcdMaskeB |= 0x04;
					MaskeB = (MaskeB & 0xE0) | 0x1B;
   					ADCON1 = 0x06;
					break;
				case 7:	// AN0..4 AN8..9
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
					TRISB     |= 0x0C;	//RB2,3
					AcdMaskeB |= 0x0C;
					MaskeB = (MaskeB & 0xE0) | 0x13;
   					ADCON1 = 0x05;
					break;
				case 8:	// AN0..4 AN8..10
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
					TRISB     |= 0x0E;	//RB2,3,1
					AcdMaskeB |= 0x0E;
					MaskeB = (MaskeB & 0xE0) | 0x11;
   					ADCON1 = 0x04;
					break;
				case 9:	// AN0..4 AN8..11
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
					TRISB     |= 0x1E;	//RB2,3,1,4
					AcdMaskeB |= 0x1E;
					MaskeB = (MaskeB & 0xE0) | 0x01;
   					ADCON1 = 0x03;
					break;
				case 10:	// AN0..4 AN8..12
					TRISA     |= 0x2F;
					AcdMaskeA |= 0x2F;
					MaskeA = (MaskeA & 0xD0) | 0x00;
					TRISB     |= 0x1F;	//RB2,3,1,4,0
					AcdMaskeB |= 0x1F;
					MaskeB = (MaskeB & 0xE0) | 0x00;
   					ADCON1 = 0x02;
					break;
			}
			switch (dataPacket._byte[3])
			{
				case 1:	ADCON1  |= 0x10; break;		//AN3 = ref+
				case 2:	ADCON1  |= 0x20; break;		//AN2 = ref-
				case 3:	ADCON1  |= 0x30; break;		//AN2 = ref-	AN3 = ref+
				default:                 break;
			}
			ADCON0=0x01; 		             // AN0, ADC-ein
			break;
		case 2: //aktuellen Eingang  0..4 8..12 festlegen
			channel = dataPacket._byte[2];
			if (channel>4) channel+=3;
			ADCON0  = (channel<<2) | 1;
			//warten
			SleepUs(40);
			break;
		case 3: //Resultat auslesen
			ADCON0bits.GO = 1;              // Start AD conversion
	   		while(ADCON0bits.NOT_DONE);     // Wait for conversion
            dataPacket._byte[2] = ADRESL;
            dataPacket._byte[3] = ADRESH;
			break;
	}
	counter = 16;
} // UP_ADC



//** F R Q *******************************************************************************//

// Frequenzmesser mit Timer0
// 0 - FRQ aus
// 1 - 
// 2 - 
// 3 - Frequenz messen manuell
// 3 - Frequenz messen autorange
void UP_FRQ(void)
{
	unsigned long int	Frequenz, Frequenz1;
	byte	FrqGrob;
	word	FrqMul;
	switch (dataPacket._byte[1])
	{
		case 0:

			break;
		case 1: 

			break;
		case 2:

			break;
		case 3: //Frequenz messen
			//RA4 auf input schalten
			TRISA  |= 0x10;
			switch (dataPacket._byte[2])
			{
				case 0:	T0CON = 0x20 ;  break;	//default	0010 0000  PS:2
				case 1:	T0CON = 0x21 ;  break;	//default	0010 0001  PS:4
				case 2:	T0CON = 0x22 ;  break;	//default	0010 0010  PS:8
				case 3:	T0CON = 0x23 ;  break;	//default	0010 0011  PS:16
				case 4:	T0CON = 0x24 ;  break;	//default	0010 0100  PS:32
				case 5:	T0CON = 0x25 ;  break;	//default	0010 0101  PS:64
				case 6:	T0CON = 0x26 ;  break;	//default	0010 0110  PS:128
				case 7:	T0CON = 0x27 ;  break;	//default	0010 0111  PS:256
				case 8:	T0CON = 0x28 ;  break;	//default	0010 1000  PS:1
			}
			TMR0H = 0;
			TMR0L = 0;
			INTCONbits.TMR0IF = 0;
			T0CONbits.TMR0ON = 1;
			//Fin_max = 6 MHz
			switch (dataPacket._byte[3])
			{
				case 0:	SleepMs(10);  break;	// default
				case 1:	SleepMs(1);   break;	// 1000 Hz	f<65 MHz
				case 2:	SleepMs(10);  break;	//  100 Hz	f<6.5 MHz
				case 3:							//   10 Hz	f<655 kHz
					SleepMs(99);  	// 100 ms
					SleepUs(1023); 
					SleepUs(12);  
 				break;	
			}
			T0CONbits.TMR0ON = 0;
            dataPacket._byte[2] = TMR0L;
            dataPacket._byte[3] = TMR0H;
			if (INTCONbits.TMR0IF) dataPacket._byte[0] = 0xFF; //Ueberlauf
			break;
		case 5: //Frequenz automatisch messen
			//RA4 auf input schalten
			TRISA  |= 0x10;
			//Grobmessung  256:1, 1ms
			T0CON = 0x27 ; 				//  PS:256
			TMR0H = 0;
			TMR0L = 0;
			INTCONbits.TMR0IF = 0;
			T0CONbits.TMR0ON = 1;
			//Fin_max = 6 MHz
			SleepMs(1);  				// 1000 Hz	f<65 MHz
			T0CONbits.TMR0ON = 0;
			// 50 MHz / 256 / 1000 = 195
			// 40 MHz / 256 / 1000 = 156  62:1
			// 20 MHz / 256 / 1000 =  78  32:1
			// 10 MHz / 256 / 1000 =  39  16:1
			//  5 MHz / 256 / 1000 =  19   8:1
			// 2.5MHz / 256 / 1000 =   9   4:1
			// 1.25MHz/ 256 / 1000 =   4   2:1
			// 0.6MHz / 256 / 1000 =   2   1:1

			// 32 MHz / 256 / 1000 = 128  62:1
			// 16 MHz / 256 / 1000 =  64  32:1
			//  8 MHz / 256 / 1000 =  32  16:1
			//  4 MHz / 256 / 1000 =  16   8:1
			//  2 MHz / 256 / 1000 =   8   4:1
			//  1 MHz / 256 / 1000 =   4   2:1
			// 0.5MHz / 256 / 1000 =   2   1:1
            FrqGrob = TMR0L;
			if (FrqGrob >156) T0CON = 0x26 ;
			else	if (FrqGrob >156) {T0CON = 0x26; FrqMul = 128;}
			else	if (FrqGrob > 78) {T0CON = 0x25; FrqMul =  64;}
			else	if (FrqGrob > 39) {T0CON = 0x24; FrqMul =  32;}
			else	if (FrqGrob > 19) {T0CON = 0x23; FrqMul =  16;}
			else	if (FrqGrob >  9) {T0CON = 0x22; FrqMul =   8;}
			else	if (FrqGrob >  4) {T0CON = 0x21; FrqMul =   4;}
			else	if (FrqGrob >  2) {T0CON = 0x20; FrqMul =   2;}
			else 					  {T0CON = 0x28; FrqMul =   1;}

			//Feinmessung  xx:1, 100ms
			TMR0H = 0;
			TMR0L = 0;
			INTCONbits.TMR0IF = 0;
			T0CONbits.TMR0ON = 1;
			//Fin_max = 6 MHz
			// 100ms
			SleepMs(99);  				// 10 Hz
			SleepUs(1023);  			// 10 Hz
			SleepUs(12);  		     	// 10 Hz
			T0CONbits.TMR0ON = 0;
            Frequenz1 = TMR0L;
            Frequenz  = TMR0H;
			Frequenz <<= 8;
			Frequenz += Frequenz1; 
            Frequenz *= 10;
            Frequenz *= FrqMul;
			dataPacket._byte[2] =  Frequenz & 0x000000FF;
			dataPacket._byte[3] = (Frequenz & 0x0000FF00) >>  8;
			dataPacket._byte[4] = (Frequenz & 0x00FF0000) >> 16;
			dataPacket._byte[5] = (Frequenz & 0xFF000000) >> 24;
			if (INTCONbits.TMR0IF) dataPacket._byte[0] = 0xFF; //Ueberlauf
			break;
	}
	counter = 16;
} // UP_FRQ



//** R S 2 3 2 **********************************************************************//

// RS232
// 8 Bit, keine Paritaet, keine Interrupts
// 1:   115200 Baud = 48 MHz / 16 / ( 25+1)  -> BRGH=1,  Error= 0.16%
// 2:    57600 Baud = 48 MHz / 16 / ( 51+1)  -> BRGH=1,  Error= 0.16%
// 3/0:  19200 Baud = 48 MHz / 16 / (155+1)  -> BRGH=1,  Error= 0.16%  standard
// 4:     9600 Baud = 48 MHz / 64 / ( 77+1)  -> BRGH=0,  Error= 0.16%
// mit Bit7 knnen signale invertiert werden

// 0 - USART aus
// 1 - initialisieren
// 2 - senden
// 3 - empfangen
// 4 - String senden
// 5 - String empfangen
// 6 -  Empfangspuffer Fuellstand
// 7 - Flush Empfangspuffer
void UP_RS232(void)
{
	byte	rsconfig;
	byte	Nummer;
	byte	Menge;
	rsconfig = USART_TX_INT_OFF & USART_RX_INT_OFF & USART_ASYNCH_MODE & USART_EIGHT_BIT & USART_CONT_RX;
	switch (dataPacket._byte[1])
	{
		case 0://USART aus
			CloseUSART();
			MaskeC |= 0xC0;
			RS232_running = 0;
			break;
		case 1://initialisieren
			switch (dataPacket._byte[2] & 0x0F)
			{
				case 0: OpenUSART(rsconfig & USART_BRGH_HIGH, 155); break;	//  19200 - standard
				case 1: OpenUSART(rsconfig & USART_BRGH_HIGH,  25); break;	// 115200
				case 2: OpenUSART(rsconfig & USART_BRGH_HIGH,  51); break;	//  57600
				case 3: OpenUSART(rsconfig & USART_BRGH_HIGH, 155); break;	//  19200
				case 4: OpenUSART(rsconfig & USART_BRGH_LOW,   77); break;	//   9600
			}
			if ((dataPacket._byte[2] & 0x80) == 0)
			{
	    		// das ist der Normalfall mit externen Treibern
				BAUDCON = BAUDCON & 0xCF;
			} else{
	    		// ohne Treiber
				BAUDCON = BAUDCON | 0x30;
			}
			MaskeC &= 0x3F;
			RS232_pointer = 0;
			RS232_running = 1;
			break;
		case 2:// 1 zeichen senden
			while(BusyUSART());						// warten auf freien Puffer
			WriteUSART(dataPacket._byte[2]);
			break;
		case 3:// 1 zeichen empfangen
			if (RS232_pointer)	
			{
				dataPacket._byte[2] = RS232_buffer[0];
				for (Nummer=0; Nummer<RS232_pointer; Nummer++)
				{
					RS232_buffer[Nummer] = RS232_buffer[Nummer+1] ;
				}
				RS232_pointer--;
				dataPacket._byte[0] = 0;			// Empfangs OK
			} else dataPacket._byte[0] = 0xFF;		// Empfangsfehler ?
			break;
		case 4://String senden
			// 0 - Klasse: RS232
			// 1 - Befehl: String senden
			// 2 - Anzahl
			// 3... daten
			for (Nummer=3; Nummer<3+dataPacket._byte[2]; Nummer++)
			{
				while(BusyUSART());					// warten auf freien Puffer
				WriteUSART(dataPacket._byte[Nummer]);
			}
			break;
		case 5://String empfangen
			if (RS232_pointer)	
			{
				Menge = dataPacket._byte[2];
				if (Menge>13) Menge = 13;
				if (Menge>RS232_pointer) Menge = RS232_pointer;
 				for (Nummer=0; Nummer<RS232_pointer; Nummer++)
				{	
					if (Nummer<Menge) dataPacket._byte[Nummer+3] = RS232_buffer[Nummer];
					RS232_buffer[Nummer] = RS232_buffer[Nummer+Menge];
				}
				RS232_pointer -= Menge;
				dataPacket._byte[2] = Menge;
			} else
			{
				dataPacket._byte[2] = 0; 
			}
			break;
		case 6:// Fuellstand
			dataPacket._byte[2] = RS232_pointer;
			break;
		case 7:// Flush Empfangspuffer
			RS232_pointer = 0;
			break;
	}
	counter = 16;
} // UP_RS232



void RS232_run(void)
{
	extern union USART USART_Status;
	if (RS232_pointer >= sizeof(RS232_buffer)) return;	// buffer full
	if (DataRdyUSART()) RS232_buffer[RS232_pointer++] = ReadUSART();

    if (USART_Status.OVERRUN_ERROR) // USART_Status.FRAME_ERROR
	{
		RCSTAbits.CREN = 0;
		RCSTAbits.CREN = 1;
	}
}	//RS232_run



//*****I 2 C ************************************************************************//

// I2C
// 0 - I2C aus
// 1 - initialisieren
// 2 - senden
// 3 - empfangen
// 4 - String senden
// 5 - String empfangen

// fuer initialisieren
// 0 - Klasse
// 1 - Befehl 1
// 2 - Mode
//		0:  Master
//		1:  Slave 7 Bit
//		2:  Slave 10 Bit
//		3:  Master
// 3 - Bustakt
//		0:  100 kHz
//		1:  400 kHz
//		2:  1 MHz

// fuer senden empfangen eines byte
// 0 - Klasse
// 1 - Befehl 2,3
// 2 - Adresse
// 3 - Daten

// fuer senden empfangen eines string
// 0 - Klasse
// 1 - Befehl 4,5
// 2 - Adresse
// 3 - Laenge
// 4..7 - Daten

//master->slave: 
//	Busbernahme
//	Adresse senden 7Bit +'0' als Schreibkennzeichen
//	warten auf ACK vom Slave
//	1 Byte senden
//	warten auf ACK vom Slave
//	1 Byte senden
//	warten auf ACK vom Slave
// ..
//	Bus freigeben

//slave->master: 
//	Busbernahme
//	Adresse senden 7Bit +'1' als Lesekennzeichen
//	warten auf ACK vom Slave
//	1 Byte empfangen
//	ACK senden
//	1 Byte empfangen
//	ACK senden
// ..
//	Bus freigeben


unsigned char I2C_ready(void)
{
//	i2c_warte 
//        btfss   PIR1, SSPIF     ; fertig? 
//        goto    i2c_warte       ; nein, noch nicht 
//        bcf     PIR1, SSPIF     ; ja, alles fertig, nun noch SSPIF zurcksetzen 
	byte	timeout;
	timeout=0;
	while (( !PIR1bits.SSPIF)  & (timeout<10))       // wait until byte received  
	{
		SleepUs(100);
		timeout++;
	}
	if (timeout >= 10) return(0xFF);// return with error
	PIR1bits.SSPIF = 0;
	return ( 0x00 );                // return with ok
}

// I2C-Bus im Master-Mode bernehmen 
void I2C_start(void)
{
//i2c_on  bcf     PIR1, SSPIF     ; SSPIF Bit lschen 
//        bsf     SSPCON2, SEN    ; Bus bernahme anweisen 
//        goto    i2c_warte 
	PIR1bits.SSPIF = 0;
	SSPCON2bits.SEN = 1;  
	I2C_ready();
}

//I2C-Bus wieder freigeben 
void I2C_stop(void)
{
//        bsf     SSPCON2, PEN    ; Bus Freigabe anweisen 
	SSPCON2bits.PEN = 1;            // initiate bus stop condition
}

//ein Byte senden 
void I2C_tx(byte daten)
{
//        movwf   SSPBUF          ; -> zum I2C-Slave bertragen 
//        goto    i2c_warte 
	//SSPBUF = daten;
	//I2C_ready();  
	WriteI2C(daten);
	IdleI2C();
}

//string empfangen fr Master oder slave
unsigned char I2C_rxs( unsigned char index, unsigned char length )
{
	while ( length -- )           // perform getcI2C() for 'length' number of bytes
	{
		dataPacket._byte[index] = getcI2C();       // save byte received
		index ++;	
		while ( SSPCON2bits.RCEN );   // check that receive sequence is over    
		if ( PIR2bits.BCLIF )         // test for bus collision
		{
			return ( -1 );            // return with Bus Collision error 
		}
		if ( length )                 // test if 'length' bytes have been read
		{
			SSPCON2bits.ACKDT = 0;    // set acknowledge bit state for ACK     
		} else {
			SSPCON2bits.ACKDT = 1;    // set not acknowledge bit  NACK     
		}
		SSPCON2bits.ACKEN = 1;        // initiate bus acknowledge sequence
		while ( SSPCON2bits.ACKEN );  // wait until ACK sequence is over        
	}
	return ( 0 );                     // last byte received so don't send ACK      
}

//string senden fr Master oder slave
signed char I2C_txs( unsigned char index, unsigned char length )
{
  while ( length -- )           // perform getcI2C() for 'length' number of bytes
  {
    if ( SSPCON1bits.SSPM3 )      // if Master transmitter then execute the following
    {
      if ( putcI2C ( dataPacket._byte[index] ) )   // write 1 byte
      {
        return ( -3 );            // return with write collision error
      }
	  index ++;
      IdleI2C();                  // test for idle condition
      if ( SSPCON2bits.ACKSTAT )  // test received ack bit state
      {
        return ( -2 );            // bus device responded with  NOT ACK
      }                           // terminate putsI2C() function
    }
    else                          // else Slave transmitter
    {
      PIR1bits.SSPIF = 0;         // reset SSPIF bit
      SSPBUF = dataPacket._byte[index],            // load SSPBUF with new data
	  index ++;
      SSPCON1bits.CKP = 1;        // release clock line 
      while ( !PIR1bits.SSPIF );  // wait until ninth clock pulse received
      if ( ( !SSPSTATbits.R_W ) && ( !SSPSTATbits.BF ) )// if R/W=0 and BF=0, NOT ACK was received
      {
        return ( -2 );            // terminate PutsI2C() function
      }
    }
  }                               // continue data writes until null character
  return ( 0 );
}

void UP_I2C(void)
{
	byte	i2cslew;
	byte	Nummer;
	switch (dataPacket._byte[1])
	{
		case 0:
			SSPCON1 &= 0xDF;                // disable synchronous serial port
			MaskeB |= 0x03; // RB0 & RB1 wieder frei
			break;
		case 1:
			SSPSTAT &= 0x3F;                // power on state 
			SSPCON1  = 0x00;                // power on state
			SSPCON2  = 0x00;                // power on state
			SSPCON1 |= 0x08;	            // master  //select serial mode 
			DDRBbits.RB1 = 1;               // Set SCL (PORTB,1) pin to input  DDRB=TRISB
			DDRBbits.RB0 = 1;               // Set SDA (PORTB,0) pin to input
			SSPCON1 |= 0x20;                // enable synchronous serial port 
			SSPADD = 0x77;					// 100kHz default
			switch (dataPacket._byte[3])	//bustakt
			{
				case 0:
					SSPSTAT |= 0x80;		// slew rate off 
					SSPADD = 0x77;			// 100kHz ; clock = 48/(4*(119+1)) = 100kHz ; 119=0x77
					break;
				case 1:
					SSPADD = 0x1D;			// 400kHz ;	clock = 48/(4*( 29+1)) = 400kHz ;  29=0x1D
					break;
				case 2:
					SSPSTAT |= 0x80;		// slew rate off 
					SSPADD = 0x0B;			//   1MHz ; clock = 48/(4*( 11+1)) = 1MHz   ;  11=0x0B
					break;
			}
			MaskeB &= 0xFC; 	// RB0 & RB1 sperren
			break;
		case 2: //senden 1 Byte 7 Bit Adresse
			I2C_start();
			I2C_tx((dataPacket._byte[2]<<1) | 0x00); //Adresse
			dataPacket._byte[0] = I2C_txs(3,1);	
			I2C_stop();
			break;
		case 3: //empfangen 1 Byte 7 Bit Adresse
			I2C_start();
			I2C_tx((dataPacket._byte[2]<<1) | 0x01); //Adresse
			dataPacket._byte[0] = I2C_rxs(2,1);							// index, Anzahl
			I2C_stop();
			break;
		case 4: //String senden 7 Bit Adresse
			// fuer senden empfangen eines string
			// 0 - Klasse
			// 1 - Befehl 4,5
			// 2 - Adresse
			// 3 - Laenge
			// 4..7 - Daten
			I2C_start();					// initiate bus start condition
			Nummer = dataPacket._byte[3];   // Anzahl der Datenbytes
			dataPacket._byte[3] = (dataPacket._byte[2]<<1);  // Adresse aufbereiten
			dataPacket._byte[0] = I2C_txs(3,Nummer+1);
			I2C_stop();						// initiate bus stop condition
			break;
		case 5: //String empfangen 7 Bit Adresse
			I2C_start();					// initiate bus start condition
			I2C_tx((dataPacket._byte[2]<<1) | 0x01);
			dataPacket._byte[0] = I2C_rxs(4,dataPacket._byte[3]);
			I2C_stop();						// initiate bus stop condition
			break;
		case 0x12: //senden 1 Byte 10 Bit Adresse
			I2C_start();
			I2C_tx(((dataPacket._byte[2] & 0x03) <<1) | 0xF0); 	// Adresse obere 2 Bit
			I2C_tx(  dataPacket._byte[3]); 						// Adresse untere 8 bit
			dataPacket._byte[0] = I2C_txs(4,1);	
			I2C_stop();
			break;
		case 0x13: //empfangen 1 Byte 10 Bit Adresse
			I2C_start();
			I2C_tx(((dataPacket._byte[2] & 0x03) <<1) | 0xF1); 	// Adresse obere 2 Bit
			I2C_tx(  dataPacket._byte[3]); 						// Adresse untere 8 bit
			dataPacket._byte[0] = I2C_rxs(2,1);					// index, Anzahl
			I2C_stop();
			break;
		case 0x14: //String senden 10 Bit Adresse
			// fuer senden empfangen eines string
			// 0 - Klasse
			// 1 - Befehl 4,5
			// 2 - AdresseH
			// 3 - AdresseL
			// 4 - Laenge
			// 5..7 - Daten
			I2C_start();														// initiate bus start condition
			Nummer = dataPacket._byte[4];   									// Anzahl der Datenbytes
			dataPacket._byte[4] =   dataPacket._byte[3];						// AdresseL aufbereiten
			dataPacket._byte[3] = ((dataPacket._byte[2] & 0x03) <<1) | 0xF0;	// AdresseH aufbereiten
			dataPacket._byte[0] = I2C_txs(3,Nummer+2);
			I2C_stop();															// initiate bus stop condition
			break;
		case 0x15: //String empfangen 10 Bit Adresse
			I2C_start();											// initiate bus start condition
			I2C_tx(((dataPacket._byte[2] & 0x03) <<1) | 0xF1); 		// Adresse obere 2 Bit
			I2C_tx(  dataPacket._byte[3]); 							// Adresse untere 8 bit
			dataPacket._byte[0] = I2C_rxs(5,dataPacket._byte[4]);
			I2C_stop();												// initiate bus stop condition
			break;
	}
	counter = 16;
} // UP_I2C


//** S P I ******************************************************************************//

// SPI
// 0 - SPI aus
// 1 - initialisieren
// 2 - senden
// 3 - empfangen
//
// RB0 - IN
// RB1 - TAKT
// RB2 - Device 1
// RB3 - Device 2
// RB4 - Device 3
// RB5 - Device 4
// RB6 - Device 5
// RB7 - Device 6
//


// ermittelt die richtige Chip-select-Leitung
byte SPI_CS(byte Nummer)
{
	byte result;
	result = 0;
	switch (Nummer & 0x0F)
	{
		case 1: result = 0x04; break;
		case 2: result = 0x08; break;
		case 3: result = 0x10; break;
		case 4: result = 0x20; break;
		case 5: result = 0x40; break;
		case 6: result = 0x80; break;
	}
	return (result);	
}


// select-Leitung low
void SPI_Slave_on(byte Nummer)
{
	LATB &= ~SPI_CS(Nummer);
}


// select-Leitung high
void SPI_Slave_off(byte Nummer)
{
	LATB |= SPI_CS(Nummer);
}



void UP_SPI(void)
{
	unsigned char sync_mode; 
	unsigned char bus_mode;
	unsigned char smp_phase;
	unsigned char spi_anzahl;
	unsigned char bytenummer;

	switch (dataPacket._byte[1])
	{
		case 0:
			// SPI aus
			CloseSPI();
			MaskeB |= 0x03; 						// RB0 & RB1 wieder frei
			MaskeC |= 0x80; 						// RC7 wieder frei
			if (SPI_SlaveMode) MaskeA |= 0x20; 		// RA5 wieder frei ## nur im Slave-Mode
			MaskeB |= SPI_Maske;					// Device-Select-Leitungen zurueckgeben
			SPI_Maske = 0;
			break;
		case 1:
			// initialisieren
			// 0 - 0x53 
			// 1 - 0x01
			// 2 - Anzahl der slaves
			// 3 - Takt
			// 4 - Mode
			// 5 - Sample
			// erst mal default
			SPI_Maske = 0;
			sync_mode = SPI_FOSC_64;		//default 187 kHz
			bus_mode  = MODE_00;
			smp_phase = SMPEND;
			SPI_SlaveMode = 0;

			// Anzahl der Slaves mit Slave-Select Leitung
			// belegen Pins am Port B ab RB2 aufsteigend
			switch (dataPacket._byte[2])
			{
				case 1: SPI_Maske = 0x04;	break;	// xxxx xX--
				case 2: SPI_Maske = 0x0C;	break;	// xxxx XX--
				case 3: SPI_Maske = 0x1C;	break;	// xxxX XX--
				case 4: SPI_Maske = 0x3C;	break;	// xxXX XX--
				case 5: SPI_Maske = 0x7C;	break;	// xXXX XX--
				case 6: SPI_Maske = 0xFC;	break;	// XXXX XX--
			}

			// Takt
			switch (dataPacket._byte[3])
			{
				case 1: sync_mode = SPI_FOSC_4;					break;		//       12 MHz
				case 2: sync_mode = SPI_FOSC_16;				break;		//        3 MHz
				case 3: sync_mode = SPI_FOSC_64;				break;		//      750 kHz
//				case 4: sync_mode = SPI_FOSC_TMR2;				break;		//
				case 5: sync_mode = SLV_SSON;SPI_SlaveMode = 1;	break;		// SPI Slave mode, /SS pin control enabled
				case 6: sync_mode = SLV_SSOFF;					break;		// SPI Slave mode, /SS pin control disabled
			}
			//aktive Taktflanke und Ruhepegel
			switch (dataPacket._byte[4])
			{
				case 1: bus_mode = MODE_00;		break;			// Setting for SPI bus Mode 0,0
				case 2: bus_mode = MODE_01;		break;			// Setting for SPI bus Mode 0,1
				case 3: bus_mode = MODE_10;		break;			// Setting for SPI bus Mode 1,0
				case 4: bus_mode = MODE_11;		break;			// Setting for SPI bus Mode 1,1
			}
			// wann wird IN_Leitung gelesen?
			switch (dataPacket._byte[5])
			{
				case 1: smp_phase = SMPEND;		break;			// Input data sample at end of data out
				case 2: smp_phase = SMPMID;		break;			// Input data sample at middle of data out
			}
			OpenSPI(sync_mode,bus_mode,smp_phase);				// init SPI-Hardware und setzt TRIS-Bits

			MaskeB &= 0xFC; 					// RB0 & RB1 sperren
			MaskeC &= 0x7F; 					// RC7 sperren
			if (SPI_SlaveMode) MaskeA &= 0xDF; 	// RA5 sperren  ## nur im Slave-Mode
			MaskeB &= ~SPI_Maske;				// Device-Select-Leitungen sperren
			TRISB  &= ~SPI_Maske;				// Device-Select-Leitungen output
			LATB   |=  SPI_Maske;				// alle Slaves deaktivieren
			break;
		case 2:
			// 1 Byte senden
			// [2] - device-Nr
			// [3] - datenbyte
			SPI_Slave_on(dataPacket._byte[2]);
			PIR1bits.SSPIF = 0;
			SSPBUF = dataPacket._byte[3];		// write byte to SSPBUF register
			if ( SSPCON1 & 0x80 )				// test if write collision occurred
				dataPacket._byte[3] = 255;		// if WCOL bit is set return 255
			 else
			{
				while( !PIR1bits.SSPIF);
				//while( !SSPSTATbits.BF );		// wait until bus cycle complete 
				dataPacket._byte[3] = 0;		// if WCOL bit is not set return non-negative#
			}
			if ((dataPacket._byte[2] & 0x40) == 0) SPI_Slave_off(dataPacket._byte[2]);
			break;
		case 3:
			// 1 Byte empfangen
			// [2] - device-Nr
			// [3] - datenbyte
			SPI_Slave_on(dataPacket._byte[2]);
			dataPacket._byte[3] = ReadSPI();
			if ((dataPacket._byte[2] & 0x40) == 0) SPI_Slave_off(dataPacket._byte[2]);
			break;
		case 4:
			// Block senden
			// [2] - device-Nr  ,
			//		bit7=0: cs ber alle byte, bit7=1 cs fr jedes bytes
			//		bit6=0: cs am Ende abschalten, bit6=1 cs am ende aktiv lassen
			// [3] - zahl der datenbytes
			// [4] - 1. datenbyte
			spi_anzahl = dataPacket._byte[3];
			dataPacket._byte[3] = 0; 					//Fehlerfrei
			for (bytenummer=4; bytenummer<(4+spi_anzahl); bytenummer++)
			{
				SPI_Slave_on(dataPacket._byte[2]);
				PIR1bits.SSPIF = 0;
				SSPBUF = dataPacket._byte[bytenummer];	// write byte to SSPBUF register
				if ( SSPCON1 & 0x80 )					// test if write collision occurred
					dataPacket._byte[3] = 255;			// Fehler: Buskollision
				 else
					while( !PIR1bits.SSPIF );			// wait until bus cycle complete 
				if (dataPacket._byte[2] >= 0x80) SPI_Slave_off(dataPacket._byte[2]);
			}
			if ((dataPacket._byte[2] & 0x40) == 0) SPI_Slave_off(dataPacket._byte[2]);
			break;
	}
	counter = 16;
} // UP_SPI




//** MWIRE ******************************************************************************//

// microwire
// 0 - SPI aus
// 1 - initialisieren (0 .. 4)
// 2 - senden
// 3 - empfangen
void UP_MWIRE(void)
{
	unsigned char high_byte;
	unsigned char low_byte;
	switch (dataPacket._byte[1])
	{
		case 0:
			// SPI aus
			CloseMwire();
			MaskeB |= 0x03; // RB0 & RB1 wieder frei
			MaskeC |= 0x80; // RC7 wieder frei
			break;
		case 1:
			// initialisieren
			//[2] mode  0..4
			switch (dataPacket._byte[2])
			{
				case 0: OpenMwire(MWIRE_FOSC_64);   break; //default 750 kHz
				case 1: OpenMwire(MWIRE_FOSC_4);    break; //       12 MHz
				case 2: OpenMwire(MWIRE_FOSC_16);   break; //        3 MHz
				case 3: OpenMwire(MWIRE_FOSC_64);   break; //      750 kHz
//				case 4: OpenMwire(MWIRE_FOSC_TMR2); break; //
			}
			MaskeB &= 0xFC; // RB0 & RB1 sperren
			MaskeC &= 0x7F; // RC7 sperren
			break;
		case 2:
			// senden
			// [2] - datenbyte
			// [3] - 1=warten auf fertig
			WriteMwire(dataPacket._byte[2]);
			if (dataPacket._byte[3] == 1) while(! DataRdyMwire() );	// busy poll
			break;
		case 3:
			// This function reads in a single byte from a Microwirex device. The Start
			// bit, opcode and address compose the high and low bytes passed into
			// this function.
			// [2] - low
			// [3] - high
			high_byte = dataPacket._byte[2];
			low_byte  = dataPacket._byte[3];
			dataPacket._byte[2] = ReadMwire(high_byte, low_byte);
			break;
	}
	counter = 16;
} // UP_MWIRE



//** Schieberegister ******************************************************************************//

// SR
// Eingang : RB0
// Ausgang : RC7
// Clock   : RB1
// 0 - SR aus
// 1 - initialisieren (0 .. 4)
// 2 - senden/empfangen
// 3 - senden/empfangen
// es werden maximal 13 Bytes gelesen
void UP_SR(void)
{
	unsigned char nummer;
	unsigned char position;
	unsigned char wert;
	unsigned char wert_r;
	switch (dataPacket._byte[1])
	{
		case 0:
			MaskeB |= 0x03; // RB0 & RB1 wieder frei
			MaskeC &= 0x80; // RC7 wieder frei
			break;
		case 1:
			// RB1 und RC7 auf Ausgang
			// Bits 0 & 1 von dataPacket._byte[2] enthalten den clock-Mode mode  0..3
			// Bit 4      von dataPacket._byte[2] enthaelt den Empfangs-Sample-Zeitpunkt
			// Bit 6 & 7  von dataPacket._byte[2] enthaelt die Taktgeschwindigkeit

			//	0 & 3: steigende Flanke vor Bitwechsel
			//	1 & 2: fallende  Flanke vor Bitwechsel
			//  0 & 2 : Takt-Ruhepegel = low
			//  1 & 3 : Takt-Ruhepegel = high
			SR_Mode   =  dataPacket._byte[2] & 0x03;
			SR_Mode_r = (dataPacket._byte[2] & 0x10) >> 4;
			switch (dataPacket._byte[2] & 0xC0)
			{
				case 0x00: SR_Delay =    10; break;	// 50 kHz
				case 0x40: SR_Delay =   100; break;	//  5 kHz
				case 0x80: SR_Delay =  1000; break;	// 500 Hz
				case 0xC0: SR_Delay = 10000; break;	//  50 Hz
			}
			MaskeB &= 0xFC; // RB0 & RB1 sperren
			MaskeC &= 0x7F; // RC7 sperren
			TRISB  &= 0xFD;	// RB1 out
			TRISB  |= 0x01;	// RB0 in
			TRISC  &= 0x7F;	// RC7 out
			if ((SR_Mode == 1) || (SR_Mode == 3)) SR_TAKT = 1; else SR_TAKT = 0;
			break;
		case 2:
		case 3:
			// [2] - Anzahl der Datenbytes
			// [3] - Datenbyte...
			// schieben mit 5 kHz,  MSB first
			for (nummer = 3; nummer < (dataPacket._byte[2]+3);  nummer++)
			{
				wert = dataPacket._byte[nummer];
				wert_r = 0;
				for (position = 0x80;  position > 0; position >>=1)
				{
					if ((SR_Mode == 0) || (SR_Mode == 3)) SR_TAKT = 1; else SR_TAKT = 0;	// end data-out time
					if (position & wert)   SR_OUT  = 1;                else SR_OUT  = 0;
					SleepUs(SR_Delay);
					if (SR_Mode_r == 1)
					{
						if (SR_IN == 1) wert_r |= position;									// sample in der mitte der data-out time
					}
					if ((SR_Mode == 0) || (SR_Mode == 3)) SR_TAKT = 0; else SR_TAKT = 1;	// start data-out time
					SleepUs(SR_Delay);
					if (SR_Mode_r == 0)
					{
						if (SR_IN == 1) wert_r |= position;									// sample am ende der data-out time
					}
				}
				dataPacket._byte[nummer] = wert_r;
			}
			if ((SR_Mode == 1) || (SR_Mode == 3)) SR_TAKT = 1; else SR_TAKT = 0;
			break;
	}
	counter = 16;
} // UP_SR



//**  L C D ****************************************************************************//

// Enable=1
void LCDE_1(void)
{
	switch (dataPacket.CMD)
	{
		case SYS_LCD1:	LCD_E1 = 1;	break;
		case SYS_LCD2:	LCD_E2 = 1;	break;
	}
	SleepUs(100);
}

// Enable=0
void LCDE_0(void)
{
	switch (dataPacket.CMD)
	{
		case SYS_LCD1:	LCD_E1 = 0;	break;
		case SYS_LCD2:	LCD_E2 = 0;	break;
	}
	SleepUs(100);
}

// warten bis das display bereit ist
void LCD_BUSY(void)
{
	BYTE puffer;
	//RB4..7 auf input
	TRISB |= 0xF0;
	LCD_RW = 1;
	LCD_RS = 0;
	SleepUs(100);
	do
	{
		LCDE_1();
		puffer._byte = PORTB;
		LCDE_0();
		LCDE_1();
		LCDE_0();
	} while (puffer.b7=1);
	//RB4..7 auf output
	TRISB &= 0x0F;
	LCD_RW = 0;
}

//8 Bit als 2x4 Bit zum LCD schreiben
void LCD_Data(byte daten)
{
	//high
	LATB = (LATB & 0x0F) | (daten & 0xF0);
	LCDE_1();
	LCDE_0();
	//low
	LATB = (LATB & 0x0F) | ((daten & 0x0F) << 4);
	LCDE_1();
	LCDE_0();
}

// zeichen schreiben
void LCD_Out(byte daten)
{
	//LCD_BUSY();
	LCD_RS = 1;
	LCD_Data(daten);
}

//steuerung
void LCD_Control(byte daten)
{
	//LCD_BUSY();
	LCD_RS = 0;
	LCD_Data(daten);
}

//steuerung
void LCD_Control_noBF(byte daten)
{
	LCD_RS = 0;
	LCD_Data(daten);
}

// LCD-Displays mit 1xHD44780-Controller
// 1x8..1x40, 2x8..2x40, 4x16..4x20
// 0 - LCD aus
// 1 - initialisieren
// 2 - schreiben
// 3 - steuern
// 4 - Block schreiben
// 5 - Block steuern
// 6 - Kursor auf zeile [2] Spalte [3]
void UP_LCD(void)
{
	byte	Nummer;
	byte	Adresse;
	switch (dataPacket._byte[1])
	{
		case 0: // ausschalten
			// [2] - zeilen (0 = 2 zeilen)
			// [3] - zeichen pro zeile (0= 16 Zeichen)
			switch (dataPacket.CMD)
			{
				case SYS_LCD1:	
					//RB0 und RB2..7 auf intput
					TRISB  |= 0xFD;
					MaskeB |= 0xFD; // RB0, RB2..RB7 wieder frei
					break;
				case SYS_LCD2:
					//RC0 und RB2..7 auf intput
					TRISC  |= 0x01;
					TRISB  |= 0xFC;			
					MaskeC |= 0x01; // RC0wieder frei
					MaskeB |= 0xFC; // RB2..RB7 wieder frei
					break;
			}
			break;
		case 1: // eonschalten, erst mal 2-zeilig
			LCD_zeilen  = dataPacket._byte[2];
			LCD_spalten = dataPacket._byte[3];
			// die meisten 1x16-Displays sind intern 2x8-Displays
			LCD_1x16 = 0;
			if ((LCD_zeilen==1)&&(LCD_spalten==16))
			{
				dataPacket._byte[2]=2;		// fuer sptere 2-zeilige initialisierung
				LCD_1x16 = 1;
			}

			SleepMs(20);
			LCDE_0();
			LCD_D4 = 0;
			LCD_D5 = 0;
			LCD_D6 = 0;
			LCD_D7 = 0;
			LCD_RS = 0;
			LCD_RW = 0;
			switch (dataPacket.CMD)
			{
				case SYS_LCD1:	
					//RB0 und RB2..7 auf output
					TRISB  &= 0x02;	 // RB0, RB2..RB7 auf output
					MaskeB &= 0x02;  // RB0, RB2..RB7 sperren
					break;
				case SYS_LCD2:
					//RC0 und RB2..7 auf output
					TRISB  &= 0x03;  // RB2..RB7 auf output
					TRISC  &= 0xFE;  // RC0 auf output
					MaskeB &= 0x03;  // RB2..RB7 sperren
					MaskeB &= 0xFE;  // RC0 sperren
					break;
			}
			LCD_D4 = 1;		// 8 bit
			LCD_D5 = 1;
			SleepUs(100);
			LCDE_1();
			LCDE_0();
			SleepMs(20);	// 0011****

			LCDE_1();		// 8 bit
			LCDE_0();
			SleepUs(100);	// 0011**** //  100+100us

			LCDE_1();		// 8 bit
			LCDE_0();
			SleepUs(100);	// 0011**** // 100+100us

			LCD_D4 = 0;		// 4 bit
			LCD_D5 = 1;
			SleepUs(100);
			LCDE_1();
			LCDE_0();		// 0010****
			
			switch	(dataPacket._byte[2])	// 1 oder zweizeilig?
			{
				case 0:	LCD_Control_noBF(0x28);	//function set, 4-bit  2-zeilig,  5x7, default
				case 1:	LCD_Control_noBF(0x20);	//function set, 4-bit  1-zeilig,  5x7
				case 2:	LCD_Control_noBF(0x28);	//function set, 4-bit  2-zeilig,  5x7
			}
			SleepUs(100);
			LCD_Control_noBF(0x08);		SleepUs(100);	//display off
			LCD_Control_noBF(0x01);		SleepMs(10);	//lschen
			LCD_Control_noBF(0x06);		SleepUs(100);	//entry mode, increment, disable display-shift
			//LCD_Control_noBF(0x0F);	SleepUs(100);	//on, kursor on, blink	//display on
			LCD_Control_noBF(0x0C);		SleepUs(100);	//on, kursor off, no-blink	//display on
			break;
		case 2: //  Zeichen schreiben
			LCD_Out(dataPacket._byte[2]);
			break;
		case 3:  // Steuern
			LCD_Control(dataPacket._byte[2]);
			if (dataPacket._byte[2]<4) SleepMs(3);		//  mindestens 1,6 ms fuer diese Befehle
			break;
		case 4: // String schreiben
			// 0 - Klasse
			// 1 - Befehl
			// 2 - Anzahl
			// 3... daten
			for (Nummer=3; Nummer<3+dataPacket._byte[2]; Nummer++)
			{
				LCD_Out(dataPacket._byte[Nummer]);
			}
			break;
		case 5: // Steuer-String
 			for (Nummer=3; Nummer<3+dataPacket._byte[2]; Nummer++)
			{
				LCD_Control(dataPacket._byte[Nummer]);
			}
			break;
		case 6: // Kursor auf zeile [2] Spalte [3]
			// 2: Zeile 0..3
			// 3: Spalte 0..39
			// Zeile 0: 0x00
			// Zeile 1: 0x40
			// Zeile 2: 0x00+Zeilenlnge
			// Zeile 3: 0x40+Zeilenlnge
			if (dataPacket._byte[2]>1) //Zeile2, 3
			{
				dataPacket._byte[3] += LCD_spalten;
			}
			Adresse = ((dataPacket._byte[2] & 0x01) * 0x40) + dataPacket._byte[3];
			if (LCD_1x16)
			{
				Adresse = dataPacket._byte[3];
				if (dataPacket._byte[3]>7)
				{
					Adresse += (0x40 - 8);
				}
			}
			LCD_Control(Adresse | 0x80);
			break;
	}
	counter = 16;
} // UP_LCD



/** P W M ********************************************************************************/

// PWM1
// 0 - PWM aus
// 1 - initialisieren (0 .. 9)
// 2 - DC einstellen  (0..1023) 
// 3 - 
void UP_PWM1(void)
{
	byte			T2_PS;
	byte			period;
	unsigned int	dutycycle;
	switch (dataPacket._byte[1])
	{
		case 0:
			ClosePWM1();
			MaskeC |= 0x04; // RC2 wieder frei
			break;
		case 1: 
			//init
			// [2] - mode 0..9
			//					1:1		4:1		16:1		periode		modes
			//  100 Stufen		480kHz	120kHz	30kHz		25			1 / 2 / 3
			//  256 Stufen		187kHz	47kHz	12kHz		64			4 / 5 / 6
			// 1024 Stufen		47kHz	12kHz	3kHz		256=0		7 / 8 / 9

			// default mode0=mode5: 47kHz, 256 Stufen
			period = 64;	 
			T2_PS = T2_PS_1_4;
			switch (dataPacket._byte[2])
			{
				case 0: period = 63;    T2_PS = T2_PS_1_4;  break;

				case 1: period = 24;    T2_PS = T2_PS_1_1;  break;
				case 2: period = 24;    T2_PS = T2_PS_1_4;  break;
				case 3: period = 24;    T2_PS = T2_PS_1_16; break;
				case 4: period = 63;    T2_PS = T2_PS_1_1;	break;
				case 5: period = 63;    T2_PS = T2_PS_1_4;	break;
				case 6: period = 63;    T2_PS = T2_PS_1_16; break;
				case 7: period = 0xFF;  T2_PS = T2_PS_1_1;	break;
				case 8: period = 0xFF;  T2_PS = T2_PS_1_4;	break;
				case 9: period = 0xFF;  T2_PS = T2_PS_1_16; break;
			}
			SetDCPWM1(1);
			OpenTimer2(TIMER_INT_OFF & T2_PS); 
			OpenPWM1(period); 					//PWM period =[(period ) + 1] x 4 x TOSC x TMR2 prescaler
			MaskeC &= 0xFB; // RC2 sperren
			break;
		case 2:
			// [2] - lower 8 bit
			// [3] - upper 2 bit
			dutycycle =   dataPacket._byte[3];
			dutycycle <<= 8;
			dutycycle |=  dataPacket._byte[2];
			SetDCPWM1(dutycycle);  // 10 Bit
			break;
	}
	counter = 16;
} // UP_PWM1




// PWM2  auf RC1
// 0 - PWM aus
// 1 - initialisieren (0 .. 9)
// 2 - DC einstellen  (0..1023) 
// 3 - 
void UP_PWM2(void)
{
	byte			T2_PS;
	byte			period;
	unsigned int	dutycycle;
	switch (dataPacket._byte[1])
	{
		case 0:
			ClosePWM2();
			MaskeC |= 0x02; // RC1 wieder frei
			break;
		case 1: 
			//init
			// [2] - mode 0..9
			//					1:1		4:1		16:1		periode		modes
			//  100 Stufen		480kHz	120kHz	30kHz		25			1 / 2 / 3
			//  256 Stufen		187kHz	47kHz	12kHz		64			4 / 5 / 6
			// 1024 Stufen		47kHz	12kHz	3kHz		256=0		7 / 8 / 9

			// default mode0=mode5: 47kHz, 256 Stufen
			period = 64;	 
			T2_PS = T2_PS_1_4;
			switch (dataPacket._byte[2])
			{
				case 0: period = 63;    T2_PS = T2_PS_1_4;  break;

				case 1: period = 24;    T2_PS = T2_PS_1_1;  break;
				case 2: period = 24;    T2_PS = T2_PS_1_4;  break;
				case 3: period = 24;    T2_PS = T2_PS_1_16; break;
				case 4: period = 63;    T2_PS = T2_PS_1_1;	break;
				case 5: period = 63;    T2_PS = T2_PS_1_4;	break;
				case 6: period = 63;    T2_PS = T2_PS_1_16; break;
				case 7: period = 0xFF;  T2_PS = T2_PS_1_1;	break;
				case 8: period = 0xFF;  T2_PS = T2_PS_1_4;	break;
				case 9: period = 0xFF;  T2_PS = T2_PS_1_16; break;
			}
			SetDCPWM2(1);
			OpenTimer2(TIMER_INT_OFF & T2_PS); 
			OpenPWM2(period); 					//PWM period =[(period ) + 1] x 4 x TOSC x TMR2 prescaler
			MaskeC &= 0xFD; // RC1 sperren
			break;
		case 2:
			// [2] - lower 8 bit
			// [3] - upper 2 bit
			dutycycle =   dataPacket._byte[3];
			dutycycle <<= 8;
			dutycycle |=  dataPacket._byte[2];
			SetDCPWM2(dutycycle);  // 10 Bit
			break;
	}
	counter = 16;
} // UP_PWM2



/** E E P R O M **************************************************************************/

// EEPROM, der interne EEPROM des PIC
// 0 - 
// 1 - 
// 2 - schreiben
// 3 - lesen
// 4 - Block schreiben
// 5 - Block lesen

// 0 - Klasse
// 1 - Befehl 2,3
// 2 - Adresse
// 3 - Daten

// 0 - Klasse
// 1 - Befehl 4,5
// 2 - Adresse
// 3 - Laenge
// 4..7 - Daten
void UP_EEPROM(void)
{
	byte	Nummer;
	word	Adresse;
	word	Ende;
	switch (dataPacket._byte[1])
	{
		case 2:
			EECON1bits.EEPGD = 0;				/* WRITE step #1 */
			EECON1bits.CFGS  = 0;
			EECON1bits.WREN = 1;				/* WRITE step #2 */
			EEADR = dataPacket._byte[2];		/* WRITE step #3 */
			EEDATA = dataPacket._byte[3];		/* WRITE step #4 */
			EECON2 = 0x55;						/* WRITE step #5 */
			EECON2 = 0xaa;						/* WRITE step #6 */
			EECON1bits.WR = 1;					/* WRITE step #7 */
			while (!PIR2bits.EEIF);				/* WRITE step #8 */
			PIR2bits.EEIF = 0;					/* WRITE step #9 */
			break;
		case 3:
			EECON1bits.EEPGD = 0;				/* READ step #1 */
			EECON1bits.CFGS  = 0;
			EEADR = dataPacket._byte[2];		/* READ step #2 */
			EECON1bits.RD = 1;					/* READ step #3 */
			dataPacket._byte[3] = EEDATA;		/* READ step #4 */
			break;
		case 4:
			Nummer = 4;
			Ende   = dataPacket._byte[2];
			Ende  += dataPacket._byte[3];
			for (Adresse=dataPacket._byte[2]; Adresse<Ende; Adresse++)
			{
				EECON1bits.EEPGD = 0;				/* WRITE step #1 */
				EECON1bits.CFGS  = 0;
				EECON1bits.WREN = 1;				/* WRITE step #2 */
				EEADR = Adresse;					/* WRITE step #3 */
				EEDATA = dataPacket._byte[Nummer];	/* WRITE step #4 */
				EECON2 = 0x55;						/* WRITE step #5 */
				EECON2 = 0xaa;						/* WRITE step #6 */
				EECON1bits.WR = 1;					/* WRITE step #7 */
				while (!PIR2bits.EEIF);				/* WRITE step #8 */
				PIR2bits.EEIF = 0;					/* WRITE step #9 */
				Nummer+=1;
			}
			break;
		case 5:
			Nummer = 4;
			Ende   = dataPacket._byte[2];
			Ende  += dataPacket._byte[3];
			for (Adresse=dataPacket._byte[2]; Adresse<Ende; Adresse++)
			{
				EECON1bits.EEPGD = 0;				/* READ step #1 */
				EECON1bits.CFGS  = 0;
				EEADR = Adresse;					/* READ step #2 */
				EECON1bits.RD = 1;					/* READ step #3 */
				dataPacket._byte[Nummer] = EEDATA;	/* READ step #4 */
				Nummer += 1;
			}
			break;
	}
	counter = 16;
} // UP_EEPROM





/** M O T O R *********************************************************************************/

// sinnvoll wre eine Beschleunigungs-/Abbrems-Funktion
// Start/Stopp-Speed z.B. 10 Hz (Periode 100)



// STEPPER-MOTOR
// 0 - step aus
// 1 - initialisieren 
// 2 - drehen
// 3 - Restschritte auslesen
// 4 - Beschleunigungswerte setzen

//Halbschritt
// step	in2	in1	A	B	C	D
//	0	+	+	-	+	-	+	0x35
//  1	+	-	-	-	-	+	0x21
//	2	+	+	+	-	-	+	0x39	
//	3	-	+	+	-	-	-	0x18
//	4	+	+	+	-	+	-	0x3A
//	5	+	-	-	-	+	-	0x22
//	6	+	+	-	+	+	-	0x36
//	7	-	+	-	+	-	-	0x14

//Vollschritt
// step	in2	in1	A	B	C	D
//	0	1	1	-	+	-	+	0x35
//	2	1	1	+	-	-	+	0x39
//	4	1	1	+	-	+	-	0x3A
//	6	1	1	-	+	+	-	0x36

//Wave
// step	in2	in1	A	B	C	D
//  1	+	-	-	-	-	+	0x21
//	3	-	+	+	-	-	-	0x18
//	5	+	-	-	-	+	-	0x22
//	7	-	+	-	+	-	-	0x14

//liefert fuer jeden Schritt die noetigen Signalpegel
byte StepPattern (byte	kanal)
{
	byte step;
	step = ste[kanal].step;
	// b4: 0- normal,			1-wave-mode
    if (ste[kanal].mode & 0x80) step |= 0x01; // ungerade Zahl fuer wave-mode
	switch (step & 0x07)
	{
		case 0:	return (0x35);	break;
		case 1:	return (0x21);	break;
		case 2:	return (0x39);  break;
		case 3:	return (0x18);	break;
		case 4:	return (0x3A);	break;
		case 5:	return (0x22);	break;
		case 6:	return (0x36);	break;
		case 7:	return (0x14);	break;
	}
} //StepPattern


// die Standard-Werte fuer die Beschleunigungstabelle eintragen
void StepBeschInit(byte kanal)
{
	ste[kanal].besch[0] = 10;
	ste[kanal].besch[1] = 8;
	ste[kanal].besch[2] = 7;
	ste[kanal].besch[3] = 6;
	ste[kanal].besch[4] = 5;
	ste[kanal].besch[5] = 4;
	ste[kanal].besch[6] = 3;
	ste[kanal].besch[7] = 2;
} //StepBeschInit


//schaltet die Ausgaenge
void	StepUniDrive(byte	muster, byte kanal)
{
	// A, B, C, D
	switch (kanal)
	{
		case 1: PORTB = (LATB & 0xF0) |  (muster & 0x0F); 							break;
		case 2: PORTB = (LATB & 0x0F) | ((muster & 0x0F) << 4);						break;
		case 3: PORTA = (LATA & 0xF0) |  (muster & 0x0F); 							break;
		case 4: PORTC = (LATC & 0xB8) |  (muster & 0x07) | ((muster<<3) & 0x40); 	break;
	}
} //StepUniDrive


// Wartezeit zwischen zwei Schritten
// periode ist byte 1..255 --> 1000Hz .. 3,9 Hz
// bei Halbschritten ist beschleunigen und Bremsen symmetrisch
// bei Vollschritten wird doppelt so stark gebremst wie beschleunigt
void StepDelay (byte kanal)
{
	int warten;
	byte starten;
	byte stoppen;
	starten = 0;
	stoppen = 0;

	warten = ste[kanal].periode;
	// b5: 0- normal,			1-beschleunigen/bremsen
	if (ste[kanal].mode & 0x20)
	{
		// beschleunigen und bremsen hier einbauen
		// 10,8,7,6,5,4,3,2	 Beschleunigungswerte
		if (ste[kanal].gone < 8)     starten = ste[kanal].besch[ste[kanal].gone];
		if (ste[kanal].schritte < 8) stoppen = ste[kanal].besch[ste[kanal].schritte];
		if (warten < starten) warten = starten;
		if (warten < stoppen) warten = stoppen;
	}
	ste[kanal].gone++;
	if (ste[kanal].mode & 0x40) SleepUs(warten*100);
	else 						SleepMs(warten);
}


// speed
//	4 Hz (250 ms)
//	10 Hz (100 ms)
//	100 Hz (10 ms)
//	1000 Hz (1 ms)
void UP_STEPuni(char kanal)
{
	switch (dataPacket._byte[1])
	{
		case 0: //off
			switch (kanal)
			{
				case 1:
					TRISB  |= 0x0F;	// RB0..RB3 input
					MaskeB |= 0x0F; // RB0..RB3 wieder frei
					break;
				case 2:
					TRISB  |= 0xF0;	// RB4..RB7 input
					MaskeB |= 0xF0; // RB4..RB7 wieder frei
					break;
				case 3:
					TRISA  |= 0x0F;	// RA0..RA3 input
					MaskeA |= 0x0F; // RA0..RA3 wieder frei
					break;
				case 4:
					TRISC  |= 0x47;	// RC0..RC2, RC6 input
					MaskeC |= 0x47; // RC0..RC2, RC6 wieder frei
					break;
			}
			break;
		case 1: //init
			StepUniDrive(StepPattern(ste[kanal].step), kanal);
			switch (kanal)
			{
				case 1:
					TRISB  &= 0xF0;	// RB0..RB3 output
					MaskeB &= 0xF0; // RB0..RB3 sperren
					break;
				case 2:
					TRISB  &= 0x0F;	// RB4..RB7 output
					MaskeB &= 0x0F; // RB4..RB7 sperren
					break;
				case 3:
					TRISA  &= 0xF0;	// RA0..RA3 output
					MaskeA &= 0xF0; // RA0..RA3 sperren
					break;
				case 4:
					TRISC  &= 0xB8;	// RC0..RC2, RC6 output
					MaskeC &= 0xB8; // RC0..RC2, RC6 sperren
					break;
			}
			break;
		case 2: // bewegen
			// [2] - lower 8 bit
			// [3] - upper 8 bit
			// [4] - 0:rechts-halb
			//		 1:links -halb
			//		 2:rechts-voll
			//		 3:links -voll
			//      b0: 0-rechts,      		1-links
			//      b1: 0-halbschritt, 		1-vollschritt
			//      b2: 0-asynchron,   		1-sofort ausfuehren
			//		b3: 0- spulen anlassen, 1-spulen abschalten
			//		b4: 0- normal,			1-wave-mode
			//		b5: 0- normal,			1-beschleunigen/bremsen
			//		b6: 0- normal,			1-10 mal schneller
			// [5] - Periode im ms (0=default= 1000 Hz)
			if (dataPacket._byte[5] == 0) dataPacket._byte[5] = 1;
			ste[kanal].running 		= 1;
			ste[kanal].mode   		= dataPacket._byte[4];
			ste[kanal].periode 		= dataPacket._byte[5];
			ste[kanal].schritte     = dataPacket._byte[3];			//high
			ste[kanal].schritte    <<= 8;
			ste[kanal].schritte    += dataPacket._byte[2];    		//low
			ste[kanal].gone			= 0;
			if ((ste[kanal].mode & 0x02) == 0x02) ste[kanal].step &= 0xFE; // gerade Zahl fuer Vollschritte

			if ((ste[kanal].mode & 0x04) == 4)	//synchron
			{
				while (ste[kanal].schritte>0)
				{
					switch (ste[kanal].mode & 0x03)	// synchrone modes
					{
						case 0: ste[kanal].step++;						break;	//rechts-halb
						case 1: ste[kanal].step--;						break;	//links -halb
						case 2: ste[kanal].step++; ste[kanal].step++;	break;	//rechts-voll
						case 3: ste[kanal].step--; ste[kanal].step--;	break;	//links -voll
					}
					StepUniDrive(StepPattern(kanal), kanal);
					StepDelay(kanal);
					ste[kanal].schritte--;
				} //while
				ste[kanal].running = 0;
				if ((ste[kanal].mode & 0x08) == 0x08) 
				{
					SleepMs(ste[kanal].periode+10); 		// warten bis rotor wirklich steht
					StepUniDrive(0, kanal);  					// spulen stromlos schalten
				}		
			} //if
			break;
		case 3:		// restschritte auslesen
			dataPacket._byte[2] =  ste[kanal].schritte & 0x00FF;	    // low
           	dataPacket._byte[3] = (ste[kanal].schritte & 0xFF00)>>8;	// high
			break;
		case 4:		// Beschleunigungstabelle setzen
			switch (dataPacket._byte[2])
			{
				case 0:	// mit Standard belegen
					StepBeschInit(kanal);
					break;
				case 2:	// frei belegen
					ste[kanal].besch[0] = dataPacket._byte[3];
					ste[kanal].besch[1] = dataPacket._byte[4];
					ste[kanal].besch[2] = dataPacket._byte[5];
					ste[kanal].besch[3] = dataPacket._byte[6];
					ste[kanal].besch[4] = dataPacket._byte[7];
					ste[kanal].besch[5] = dataPacket._byte[8];
					ste[kanal].besch[6] = dataPacket._byte[9];
					ste[kanal].besch[7] = dataPacket._byte[10];
				case 3:	// auslesen
					dataPacket._byte[3] = ste[kanal].besch[0];
					dataPacket._byte[4] = ste[kanal].besch[1];
					dataPacket._byte[5] = ste[kanal].besch[2];
					dataPacket._byte[6] = ste[kanal].besch[3];
					dataPacket._byte[7] = ste[kanal].besch[4];
					dataPacket._byte[8] = ste[kanal].besch[5];
					dataPacket._byte[9] = ste[kanal].besch[6];
					dataPacket._byte[10] =ste[kanal].besch[7];
					break;
			}
			break;
	}
	counter = 16;
} // UP_STEPuni



void stepUnirun (char kanal)
{
	if (ste[kanal].schritte>0)
	{
		switch (ste[kanal].mode & 0x03)	// asynchrone modes
		{
			case 0: ste[kanal].step++;						break;
			case 1: ste[kanal].step--;						break;
			case 2: ste[kanal].step++; ste[kanal].step++;	break;
			case 3: ste[kanal].step--; ste[kanal].step--;	break;
		}
		StepUniDrive(StepPattern(kanal), kanal);
		StepDelay(kanal);
		ste[kanal].schritte--;
	}
	if (ste[kanal].schritte <= 0)
	{
		ste[kanal].running = 0;
		if ((ste[kanal].mode & 0x08) == 0x08) 
		{
			SleepMs(ste[kanal].periode+10); 		// warten bis Rotor wirklich steht
			StepUniDrive(0, kanal);  				// Spulen stromlos schalten
		}	
	}
}	// stepUnirun



/** L 2 9 7 - M O T O R *********************************************************************************/

// STEPPER-MOTOR
// 0 - step aus
// 1 - initialisieren 
// 2 - drehen
// 3 - Restschritte auslesen

// speed
//	4 Hz (250 ms)
//	10 Hz (100 ms)
//	100 Hz (10 ms)
//	1000 Hz (1 ms)

// Kanal 1	RB0	RB1
// Kanal 2	RB2	RB3
// Kanal 3	RB4	RB5
// Kanal 4	RB6	RB7

void UP_L297_Uni(char kanal)
{
	int	schritte;
	switch (dataPacket._byte[1])
	{
		case 0: //off
			switch (kanal)
			{
				case 1:
					TRISB  |= 0x03;	// RB0..RB1 input
					MaskeB |= 0x03; // RB0..RB1 wieder frei
					break;
				case 2:
					TRISB  |= 0x0C;	// RB2..RB3 input
					MaskeB |= 0x0C; // RB2..RB3 wieder frei
					break;
				case 3:
					TRISB  |= 0x30;	// RB4..RB5 input
					MaskeB |= 0x30; // RB4..RB5 wieder frei
					break;
				case 4:
					TRISB  |= 0xC0;	// RB6..RB7 input
					MaskeB |= 0xC0; // RB6..RB7 wieder frei
					break;
			}
			break;
		case 1: //init
			switch (kanal)
			{
				case 1:
					TRISB  &= 0xFC;	// RB0..RB1 output
					MaskeB &= 0xFC; // RB0..RB1 sperren
					break;
				case 2:
					TRISB  &= 0xF3;	// RB2..RB3 output
					MaskeB &= 0xF3; // RB2..RB3 sperren
					break;
				case 3:
					TRISB  &= 0xCF;	// RB4..RB5 output
					MaskeB &= 0xCF; // RB4..RB5 sperren
					break;
				case 4:
					TRISB  &= 0x3F;	// RB6..RB7 output
					MaskeB &= 0x3F; // RB6..RB7 sperren
					break;
			}
			break;
		case 2: // bewegen
			// [2] - lower 8 bit der Schrittzahl
			// [3] - upper 8 bit der Schrittzahl
			// [4] - 0:rechts
			//		 1:links
			//      b0: 0-rechts,      1-links
			//      b2: 0-asynchron,   1-sofort ausfuehren
			// [5] - Periode im ms (0=default= 1000 Hz)
			if (dataPacket._byte[5] == 0) dataPacket._byte[5] = 1; //defaul 1000 Hz
			L297[kanal].running    = 1;
			L297[kanal].mode._byte = dataPacket._byte[4];
			L297[kanal].periode    = dataPacket._byte[5];
			L297[kanal].schritte   = dataPacket._byte[3];			// high
			L297[kanal].schritte  <<= 8; 
			L297[kanal].schritte  += dataPacket._byte[2];    		// low
			switch (kanal)	// Einstellung der Drehrichtung
			{
				case 1:	LATBbits.LATB1 = L297[kanal].mode.b0; break;	// CW/CCW 1
				case 2:	LATBbits.LATB3 = L297[kanal].mode.b0; break;	// CW/CCW 2
				case 3:	LATBbits.LATB5 = L297[kanal].mode.b0; break;	// CW/CCW 3
				case 4:	LATBbits.LATB7 = L297[kanal].mode.b0; break;	// CW/CCW
			}
			if ((L297[kanal].mode._byte & 0x04) == 4)	//synchron
			{
				while (L297[kanal].schritte > 0)
				{
					switch (kanal)
					{
						// das sind in der Paxis 5us lange low-pulse
						case 1:
							LATBbits.LATB0   = 0;
							SleepUs(1);
							LATBbits.LATB0   = 1;
							break;
						case 2:
							LATBbits.LATB2   = 0;
							SleepUs(1);
							LATBbits.LATB2   = 1;
							break;
						case 3:
							LATBbits.LATB4   = 0;
							SleepUs(1);
							LATBbits.LATB4   = 1;
							break;
						case 4:
							LATBbits.LATB6   = 0;
							SleepUs(1);
							LATBbits.LATB6   = 1;
							break;
					}
					SleepMs(L297[kanal].periode);
					L297[kanal].schritte--;
				} //while
				L297[kanal].running = 0;
			} //if
			break;
		case 3:
            dataPacket._byte[2] =  L297[kanal].schritte & 0x00FF;	    // low
            dataPacket._byte[3] = (L297[kanal].schritte & 0xFF00)>>8;	// high
			break;
	}
	counter = 16;
} // UP_L297_Uni


void L297_Unirun(char kanal)
{
	if (L297[kanal].schritte>0)
	{
		switch (kanal)
		{
			case 1:
				LATBbits.LATB0   = 0;
				SleepUs(1);
				LATBbits.LATB0   = 1;
				break;
			case 2:
				LATBbits.LATB2   = 0;
				SleepUs(1);
				LATBbits.LATB2   = 1;
				break;
			case 3:
				LATBbits.LATB4   = 0;
				SleepUs(1);
				LATBbits.LATB4   = 1;
				break;
			case 4:
				LATBbits.LATB6   = 0;
				SleepUs(1);
				LATBbits.LATB6   = 1;
				break;
		}
		SleepMs(L297[kanal].periode);
		L297[kanal].schritte--;
	}
	if (L297[kanal].schritte <= 0)  L297[kanal].running = 0;
}	// L297_Unirun



/** S E R V O S *********************************************************************************/

// SERVOs
// 0 - aus
// 1 - initialisieren (2: Maske, 1=on, 0=off)
// 2 - DC einstellen  (2..9:  DC=0..99) 
// 3 - Nullpunkt verndern
// 1000 ... 2000 us Pulse an RB0..RB7

void UP_ServoB(void)
{
	switch (dataPacket._byte[1])
	{
		case 0: //off
			ServoBMaske._byte = 0;
			ServoB_running    = 0;
			ServoBPuls[0] = 50;  
			ServoBPuls[1] = 50;  
			ServoBPuls[2] = 50;  
			ServoBPuls[3] = 50;  
			ServoBPuls[4] = 50;    
			ServoBPuls[5] = 50;  
			ServoBPuls[6] = 50;  
			ServoBPuls[7] = 50;   
			break;
		case 1: //init
			ServoBMaske._byte = dataPacket._byte[2]; 
			ServoB_running    = (ServoBMaske._byte != 0);
			break;
		case 2: //DC
			ServoBPuls[0] = dataPacket._byte[2];  
			ServoBPuls[1] = dataPacket._byte[3];  
			ServoBPuls[2] = dataPacket._byte[4];  
			ServoBPuls[3] = dataPacket._byte[5];  
			ServoBPuls[4] = dataPacket._byte[6];  
			ServoBPuls[5] = dataPacket._byte[7];  
			ServoBPuls[6] = dataPacket._byte[8];  
			ServoBPuls[7] = dataPacket._byte[9];  
			break;
		case 3: //nullpunkt
			ServoNull = (word)dataPacket._byte[2]*5; 
			break;
	}
	counter = 16;
} //UP_ServoB



void UP_ServoC(void)
{
	switch (dataPacket._byte[1])
	{
		case 0: //off
			ServoCMaske._byte = 0;
			ServoC_running    = 0;
			ServoCPuls[0] = 50;  
			ServoCPuls[1] = 50;  
			ServoCPuls[2] = 50;  
			ServoCPuls[3] = 50;  
			ServoCPuls[4] = 50;    
			ServoCPuls[5] = 50;  
			ServoCPuls[6] = 50;  
			ServoCPuls[7] = 50;   
			break;
		case 1: //init
			ServoCMaske._byte = dataPacket._byte[2]; 
			ServoC_running    = (ServoCMaske._byte != 0);
			break;
		case 2: //DC
			ServoCPuls[0] = dataPacket._byte[2];  
			ServoCPuls[1] = dataPacket._byte[3];  
			ServoCPuls[2] = dataPacket._byte[4];  
			ServoCPuls[3] = dataPacket._byte[5];  
			ServoCPuls[4] = dataPacket._byte[6];  
			ServoCPuls[5] = dataPacket._byte[7];  
			ServoCPuls[6] = dataPacket._byte[8];  
			ServoCPuls[7] = dataPacket._byte[9];  
			break;
		case 3: //nullpunkt
			ServoNull = (word)dataPacket._byte[2]*5; 
			break;
	}
	counter = 16;
} //UP_ServoC


// 50 Hz = 20ms
// 8 x 2ms = 16ms
void ServoB_run (void)
{
		word pause = 2000;		// x 10us = 20ms
		if (ServoBMaske.b0) {TRISBbits.TRISB0 = 0; LATBbits.LATB0 = 1; SleepUs(ServoNull + ((word)ServoBPuls[0]*10)); pause-=ServoBPuls[0]; pause-=100; LATBbits.LATB0 = 0;}
		if (ServoBMaske.b1) {TRISBbits.TRISB1 = 0; LATBbits.LATB1 = 1; SleepUs(ServoNull + ((word)ServoBPuls[1]*10)); pause-=ServoBPuls[1]; pause-=100; LATBbits.LATB1 = 0;}
		if (ServoBMaske.b2) {TRISBbits.TRISB2 = 0; LATBbits.LATB2 = 1; SleepUs(ServoNull + ((word)ServoBPuls[2]*10)); pause-=ServoBPuls[2]; pause-=100; LATBbits.LATB2 = 0;}
		if (ServoBMaske.b3) {TRISBbits.TRISB3 = 0; LATBbits.LATB3 = 1; SleepUs(ServoNull + ((word)ServoBPuls[3]*10)); pause-=ServoBPuls[3]; pause-=100; LATBbits.LATB3 = 0;}
		if (ServoBMaske.b4) {TRISBbits.TRISB4 = 0; LATBbits.LATB4 = 1; SleepUs(ServoNull + ((word)ServoBPuls[4]*10)); pause-=ServoBPuls[4]; pause-=100; LATBbits.LATB4 = 0;}
		if (ServoBMaske.b5) {TRISBbits.TRISB5 = 0; LATBbits.LATB5 = 1; SleepUs(ServoNull + ((word)ServoBPuls[5]*10)); pause-=ServoBPuls[5]; pause-=100; LATBbits.LATB5 = 0;}
		if (ServoBMaske.b6) {TRISBbits.TRISB6 = 0; LATBbits.LATB6 = 1; SleepUs(ServoNull + ((word)ServoBPuls[6]*10)); pause-=ServoBPuls[6]; pause-=100; LATBbits.LATB6 = 0;}
		if (ServoBMaske.b7) {TRISBbits.TRISB7 = 0; LATBbits.LATB7 = 1; SleepUs(ServoNull + ((word)ServoBPuls[7]*10)); pause-=ServoBPuls[7]; pause-=100; LATBbits.LATB7 = 0;}
		if (pause>0) SleepUs(pause * 10);
} //ServoB_run


// 80 Hz = 12.5ms
// 5 x 2ms = 10ms
void ServoC_run (void)
{
		word pause = 1250;		// x 10us = 12.5ms
		if (ServoCMaske.b0) {TRISCbits.TRISC0 = 0; LATCbits.LATC0 = 1; SleepUs(ServoNull + ((word)ServoCPuls[0]*10)); pause-=ServoCPuls[0]; pause-=100; LATCbits.LATC0 = 0;}
		if (ServoCMaske.b1) {TRISCbits.TRISC1 = 0; LATCbits.LATC1 = 1; SleepUs(ServoNull + ((word)ServoCPuls[1]*10)); pause-=ServoCPuls[1]; pause-=100; LATCbits.LATC1 = 0;}
		if (ServoCMaske.b2) {TRISCbits.TRISC2 = 0; LATCbits.LATC2 = 1; SleepUs(ServoNull + ((word)ServoCPuls[2]*10)); pause-=ServoCPuls[2]; pause-=100; LATCbits.LATC2 = 0;}
//		if (ServoCMaske.b3) {TRISCbits.TRISC3 = 0; LATCbits.LATC3 = 1; SleepUs(ServoNull + ((word)ServoCPuls[3]*10)); pause-=ServoCPuls[3]; pause-=100; LATCbits.LATC3 = 0;}	// kein RC3 beim 18F2455
//		if (ServoCMaske.b4) {TRISCbits.TRISC4 = 0; LATCbits.LATC4 = 1; SleepUs(ServoNull + ((word)ServoCPuls[4]*10)); pause-=ServoCPuls[4]; pause-=100; LATCbits.LATC4 = 0;}	// kein RC4 beim 18F2455
//		if (ServoCMaske.b5) {TRISCbits.TRISC5 = 0; LATCbits.LATC5 = 1; SleepUs(ServoNull + ((word)ServoCPuls[5]*10)); pause-=ServoCPuls[5]; pause-=100; LATCbits.LATC5 = 0;}	// kein RC5 beim 18F2455
		if (ServoCMaske.b6) {TRISCbits.TRISC6 = 0; LATCbits.LATC6 = 1; SleepUs(ServoNull + ((word)ServoCPuls[6]*10)); pause-=ServoCPuls[6]; pause-=100; LATCbits.LATC6 = 0;}
		if (ServoCMaske.b7) {TRISCbits.TRISC7 = 0; LATCbits.LATC7 = 1; SleepUs(ServoNull + ((word)ServoCPuls[7]*10)); pause-=ServoCPuls[7]; pause-=100; LATCbits.LATC7 = 0;}
		if (pause>0) SleepUs(pause * 10);
} //ServoC_run



/** C O U N T E R *********************************************************************************/

// benutze timer 0
// 0 - aus
// 1 - initialisieren
// 2 - auslesen
// 3 - setzen
// 4 - ruecksetzen

//timer 0 zaehlt pulse an RA4
void UP_Counter0(void)
{
	WORD Timer0Wert;
	switch (dataPacket._byte[1])
	{
		case 0: //off
			CloseTimer0();
			MaskeA   |= 0x10;	// RA4 freigeben		
			break;
		case 1: //init
			OpenTimer0( TIMER_INT_OFF &
						T0_16BIT &
						T0_SOURCE_EXT &
						T0_PS_1_1  );
			WriteTimer0(0);
			TRISA  |= 0x10;	// RA4 input
 			MaskeA &= 0xEF; // RA4 sperren
			break;
		case 2: //auslesen
			Timer0Wert._word = ReadTimer0();
            dataPacket._byte[2] = Timer0Wert.byte0;   // low
            dataPacket._byte[3] = Timer0Wert.byte1;	  // high
			break;
		case 3: //setzen
            Timer0Wert.byte0 = dataPacket._byte[2];   // low
            Timer0Wert.byte1 = dataPacket._byte[3];	  // high
			WriteTimer0(Timer0Wert._word);
			break;
		case 4: //ruecksetzen
			WriteTimer0(0);
			break;
	}
	counter = 16;
} //UP_Counter0


// Timer 3 zaehlt Pulse an RC0
void UP_Counter3(void)
{
	WORD Timer3Wert;
	switch (dataPacket._byte[1])
	{
		case 0: //off
			CloseTimer3();
			MaskeC   |= 0x01;	// RC0 freigeben
			break;
		case 1: //init
			OpenTimer3( TIMER_INT_OFF &
						T3_16BIT_RW &
						T3_SOURCE_EXT &
						T3_PS_1_1 &
						T1_SOURCE_CCP &
						T3_SYNC_EXT_OFF );
			T1CONbits.T1OSCEN = 0;	// ansonsten gehen Pin RC0 und RC1 auf input, RC1 ist aber berflssig
			WriteTimer3(0);
			TRISC  |= 0x01;	// RC0 input
 			MaskeC &= 0xFE; // RC0 sperren
			break;
		case 2: //auslesen
			Timer3Wert._word = ReadTimer3();
            dataPacket._byte[2] = Timer3Wert.byte0;   // low
            dataPacket._byte[3] = Timer3Wert.byte1;	  // high
			break;
		case 3: //setzen
            Timer3Wert.byte0 = dataPacket._byte[2];   // low
            Timer3Wert.byte1 = dataPacket._byte[3];	  // high
			WriteTimer3(Timer3Wert._word);
			break;
		case 4: //ruecksetzen
			WriteTimer3(0);
			break;
	}
	counter = 16;
} //UP_Counter3


/****  W A R T E S C H L E I F E N **************************************************************/

// verzgerungen bis max. 63 ms
// Resonator 20 MHz, CPU-Takt 48 MHz, Cycl. 12 MHz
// 1 us = 12 Zyklen
// 1 ms = 12*1000 Zyklen
// High-Teil x 3072
// Low Teil x 12
// die delay-Routinen vertragen als input nur  1..255
void SleepUs(unsigned int us)
{
	byte teilzeit;
	teilzeit = us & 0x00FF;
    if (teilzeit>0)
	{
		Delay10TCYx(teilzeit);		// 10 Zyklen ~ 1s
	}
	teilzeit = us >> 8;			// 1/256
    if (teilzeit>0)
	{
		Delay1KTCYx(teilzeit);	// 3070 Zyklen ~258s ~0,25ms
		Delay1KTCYx(teilzeit);
		Delay1KTCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
	}
} //SleepUs



// verzgerungen bis max. 255 ms
// Resonator 20 MHz, CPU-Takt 48 MHz, Cycl. 12 MHz
// 1 us = 12 Zyklen
// 1 ms = 12*1000 Zyklen
// 4 ms = 48x1000 Zyklen
// die delay-Routinen vertragen als input nur  1..255
void SleepMs(unsigned int ms)
{
    if (ms>0)
	{
		Delay10KTCYx(ms);
		Delay1KTCYx(ms);
		Delay1KTCYx(ms);
	}
} //SleepMs



/****  E E P R O M   F U N K T I O N E N  F U E R   S T E U E R P I C ********************/


//lesen von Daten aus dem EEPROM des Steuerpic
//0 - Befehl
//1 - Startadresse -low
//2 - Startadresse high = 0
//3 - Blocklnge
void readEdata(void)
{
	byte	Nummer;
	word	Adresse;
	dataPacket._byte[0] = READ_EDATA;
	counter =63;
	Nummer = 4;
	for (Adresse=dataPacket._byte[1]; Adresse<dataPacket._byte[1]+dataPacket._byte[3]; Adresse++)
	{
		EECON1bits.EEPGD = 0;				/* READ step #1 */
		EEADR = Adresse;					/* READ step #2 */
		EECON1bits.RD = 1;					/* READ step #3 */
		dataPacket._byte[Nummer] = EEDATA;	/* READ step #4 */
		Nummer += 1;
	}
	/**** EEPROM read
	MOVLW	DATA_EE_ADDR 	;
	MOVWF	EEADR 			; Lower bits of Data Memory Address to read
	BCF		EECON1, EEPGD 	; Point to DATA memory
	BCF		EECON1, CFGS 	; Access EEPROM
	BSF		EECON1, RD 		; EEPROM Read
	MOVF	EEDATA, W 		; W = EEDATA
	*********/
}


//lesen von Daten aus dem EEPROM des Steuerpic
//0 - Befehl
//1 - Startadresse -low
//2 - Startadresse high = 0
//3 - Blocklnge
//4 - 1. Datenbyte
void writeEdata(void)
{
	byte	Nummer;
	word	Adresse;
	counter = 0x01;
	dataPacket._byte[0] = WRITE_EDATA;
	Nummer = 4;
	for (Adresse=dataPacket._byte[1]; Adresse<dataPacket._byte[1]+dataPacket._byte[3]; Adresse++)
	{
		EECON1bits.EEPGD = 0;				/* WRITE step #1 */
		EECON1bits.WREN = 1;				/* WRITE step #2 */
		EEADR = Adresse;					/* WRITE step #3 */
		EEDATA = dataPacket._byte[Nummer];	/* WRITE step #4 */
		EECON2 = 0x55;						/* WRITE step #5 */
		EECON2 = 0xaa;						/* WRITE step #6 */
		EECON1bits.WR = 1;					/* WRITE step #7 */
		while (!PIR2bits.EEIF)				/* WRITE step #8 */
			;
		PIR2bits.EEIF = 0;					/* WRITE step #9 */
		Nummer+=1;
	}
}

	/**** EEPROM write
	MOVLW 	DATA_EE_ADDR 	;
	MOVWF 	EEADR 			; Lower bits of Data Memory Address to write
	MOVLW 	DATA_EE_DATA 	;
	MOVWF 	EEDATA 			; Data Memory Value to write
	BCF 	EECON1, EPGD 	; Point to DATA memory
	BCF 	EECON1, CFGS 	; Access EEPROM
	BSF 	EECON1, WREN 	; Enable writes
	BCF 	INTCON, GIE 	; Disable Interrupts
	MOVLW 	55h ;

	;Required Sequence
	MOVWF 	EECON2 			; Write 55h
	MOVLW 	0AAh 			;
	MOVWF 	EECON2 			; Write 0AAh
	BSF 	EECON1, WR 		; Set WR bit to begin write

	BSF 	INTCON, GIE 	; Enable Interrupts
							; User code execution
	BCF 	EECON1, WREN 	; Disable writes on write complete (EEIF set)
	**************/



/****  A D C   F U N K T I O N E N  ********************************************************/

// Spannung an AN1 auslesen *************************
// sollte mit CCP1 synchronisiert werden
void ReadADC(void)
{
	ADCON0bits.GO = 1;              // Start AD conversion
   	while(ADCON0bits.NOT_DONE);     // Wait for conversion
    return;
}//end ReadADC



/****  P W M   F U N K T I O N E N ********************************************************/

// PWM initialisieren*************************
// 100kHz mit Timer2=PR2 bei CPU=48MHz, Cycl=12 MHz
// Timer Prescaler 1, 120 Takte -> PR2=D'120'=0x78
// RC2, CCP1 output

//13V   0x29
void initPWM(void)
{
    TRISC  &= 0xFB;			// RC2 output
	PR2     = 0x78;			// 120 -> 100 kHz (10 s)
	CCP1CON = 0x0C;			// PWM-Mode
	CCPR1L  = 0x0C;			// 12 -> 10%
    T2CON   = 4;            // Timer2 an PreScaler=1
    return;
}//end initPWM



/** EOF user.c ***************************************************************/
