/*********************************************************************
 * FileName:        user.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Company:         sprut
 * Copyright:       2007-2010 Joerg Bredendiek (sprut)
 *
 *
 ********************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



/*** Pinbelegung*************************************************************
*
*	PORTA	0	Comp-Eingang (-) (mit RA2 verbinden)
*			1	ADC-Eingang
*			2	Uref-Ausgang (mit RA0 verbinden)
*			3	Comp-Eingang (+) mit miniadsb verbinden
*			4	Comp-Ausgang -> RC0
*
*	PORTB	0	SWITCH0 remote	0=remote 1=local
*			1	SWITCH1 port	0=RS232  1=USB
*			2	SWITCH2 tag		0=Zeitmarke an
*			3	PWM2 Ausgang
*			4	SWITCH3 DF17	0=nur DF17
*			5	SWITCH4 CRC		0=nur DF17 mit CRC
*			6	SWITCH4 1MBit	0=1MBit 1=115kBit	
*			7	TEST-LED
*
*	PORTC	0	TTL-Eingang (mit PA4 verbinden)
*			1	LED HDR
*			2	LED	FRM
*			4	USB D-
*			5	USB	D+
*
*****************************************************************************/


/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include <usart.h>
#include "system\typedefs.h"
#include "system\usb\usb.h"
#include "io_cfg.h"             // I/O pin mapping

#include "delays.h"
//#include "system\interrupt\interrupt.h"
#include "user\user.h"
#include "user\adsbin.h"
#include "user\rs232.h"
#include "user\agc.h"

#include "timers.h"
#include "pwm.h"


/** V A R I A B L E S ********************************************************/
#pragma udata

byte	UsbSelected		= 0;		// usb oder rs232 ?

byte	RS232_rx_count	= 0;		// zahl der per RS232 empfangenen bytes
byte	RS232_rx_EOL	= 0;

char	input_buffer[64];			// 64 byte							100..140
byte	rgetascii;
byte 	hb_counter 		= 0;
byte	ib_counter 		= 0;
char	asciiKor   		= 0;
byte	befehl     		= 0;		//eine gueltige Befehlszeile wurde empfangen
byte	wertgueltig  	= 0; 		//steht in rgetascii schon was?
byte	zeileangefangen = 0; 		//kam schon mal ein #?



byte			counter	= 0;		// anzahl der zu sendenden antwort-bytes
byte			counter2= 0;		// anzahl der zu sendenden adsb-bytes
byte			c2Offset= 0;
byte			Limit;
DATA_PACKET 	dataPacket;			// 20 byte							143...183

byte			adsb[14];			// 14 byte fuer den aktuell zu empfangenen Frame
byte			Flags;
byte			Zeit;
byte			Lang;
byte			adsbLoopL;
byte			adsbLoopH;
byte			Count_Error_Data;
byte			Count_Error_Hdr;
WORD			Sum_Error_Data;
WORD			Sum_Error_Hdr;
DWORD			CRC;
byte			ADSB_Mode; 
byte			DF;					// Downloadformat

BYTE			Time;				// puffer
DWORD			TimeL;				// untere 2 Byte des Zeitcode + 2 Bytes extra
BYTE			Time0;				// unterstes Byte des Zeitcode
BYTE			Time1;				// zweites Byte des Zeitcodes
DWORD			TimeU;				// obere 4 byte des Zeitcodes

DWORD			FrameNr;

byte			TimeCode    = 0;
byte			SendFrameNr = 0;
byte			SendHeartBeat=0;

byte			T2_PS;
byte			pwm_period;
unsigned int	pwm_near_dc;
unsigned int	pwm_dutycycle;
WORD			U_pegel;			//HF-Spannung
WORD			U_ref;				//Referenzspannung

byte			agc_offset  = 100; 	// ref soll 100mV ueber pegel sein
byte			agc_mode    = 1;		// 1-on   0-off
byte			agc_counter = 1;
byte			near_far_mode    = 0;
byte			near_far_counter = 0;

char			ASCII[17] = "0123456789ABCDEF";


#pragma udata test=0x0200

char	adsb_buffer[adsb_buffer_size];		// 256 byte

//#pragma udata test2=0x0300

//char	adsb_buffer2[100];		// xx byte

//stack ist 0x100 lang 0x300 ... 0x3FF


/** D E C L A R A T I O N S **************************************************/
#pragma code
//Aufruf nach Reset
void UserInit(void)
{
	// alles analoge initialisieren
	agc_init();

	// PWM fuer Referenzspannungserzeugung
	// belegt Timer2
	pwm_init();

	//Timer 0: interne Uhr, Schritt = 21.33333 us = 64us/3
	// verwendet fuer Comparatorreferenzvoltage
	TMR0H = 0;
	TMR0L = 0;
	T0CON = 0x87;				//timer0, 16 bit, 12MHz/256 = 46875 Hz, Ueberlauf nach 1.3981 s
    INTCONbits.TMR0IF 	= 0;

	//Timer 1: interne Uhr, Schritt = 0.0833 us*8 = 1/12 MHz *8
	// verwendet fuer Zeitmarken
	TMR1H = 0;
	TMR1L = 0;
	T1CON = 0xB1;				//timer1, 16 bit, 12MHz/8, Ueberlauf nach 43 ms
    PIR1bits.TMR1IF 	= 0;

	Time0._byte			= 0;
	Time1._byte			= 0;
	TimeL._dword		= 0;
	TimeU._dword 		= 0;
	TimeCode   			= 0;	// Zeitmarken senden?
	SendFrameNr			= 0;
	FrameNr._dword 		= 0;

	// init pins
	//auch fuer RS232 muessen TX (RC6) und RX RC7) auf input eingestellt sein !
	TRISA = 0xEF;				// RA4 ist output fuer Comparator
	TRISB = 0x77;				// RB7 Test-LED, RB3-PWM		0111 0111
	TRISC = 0xF9;				// RC1, RC2 -output for LEDs
	INTCON2bits.NOT_RBPU = 0;	// pull-up fuer PORTB einschalten
	PORTA = 0;
	PORTB = 0;
	PORTC = 0;

	CRC._dword 		= 0;
	ADSB_Mode  		= 0;
	U_pegel._word 	= 0;
	U_ref._word   	= 0;
	Sum_Error_Hdr._word	= 0;
	Sum_Error_Hdr._word	= 0;

	RS232_rx_count	= 0;
	RS232_rx_EOL	= 0;
	counter			= 0;		// anzahl der zu sendenden antwort-bytes
	counter2		= 0;		// anzahl der zu sendenden adsb-bytes
	c2Offset		= 0;
}//end UserInit




/******************************************************************************
 * Function:        void ProcessIO(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is a place holder for other user routines.
 *                  It is a mixture of both USB and non-USB tasks.
 *
 * Note:            None
 *****************************************************************************/
void ProcessIO(void)
{   
	if (UsbSelected)
	{
    	// User Application USB tasks
    	if((usb_device_state < CONFIGURED_STATE)||(UCONbits.SUSPND==1)) return;
	}
    ServiceRequests();
}//end ProcessIO



// A S C I I *****************************************************************************


// ermittelt, ob es sich bei einem Zeichen um ein Trennzeichen handelt
// ermittle den noetigen Wert fuer asciiKor
// alles ausser 0..9/a..f/A.F sind trennzeichen
// aus A..F/a..F wird 10..15 (0x0A..0x0F)
// input: zeichen
// output:	-1 : Trennzeichen
//			-2 : Zeilenende
//			-3 : Zeilenanfang
//			 0 : ASCII-Zeichen
byte	trennzeichen(char	zeichen)
{
	if (zeichen== '#') return -3; //SOL
	if (zeichen==0x00) return -2; //EOL
	if (zeichen==0x0A) return -2; //EOL
	if (zeichen==0x0D) return -2; //EOL

	asciiKor = 0x00;
	if (zeichen < '0') return -1;	//   .. /
	if (zeichen <='9') 				// 0 .. 9
	{
		asciiKor = '0';
		return 0;
	}
	if (zeichen < 'A') return -1;	// : .. @
	if (zeichen <='F') 				// A .. F
	{
		asciiKor = 'A'-10;
		return 0;
	}
	if (zeichen < 'a') return -1;	// G .. '
	if (zeichen <='f') 				// a .. f
	{
		asciiKor = 'a'-10;
		return 0;
	}
	return -1;						// g .. 
} //trennzeichen


// input_buffer[ib_counter] steht auf letztem Trennzeichen
// input_buffer lesen bis zum nchsten Trennzeichen
// wandeln in Hex -> rgetascii
// bei Zeilenende  -1
byte	getascii(void)
{
	rgetascii = 0;
	while (trennzeichen(input_buffer[ib_counter]) == -1) ib_counter++;  // trennzeichen berspringen
	if    (trennzeichen(input_buffer[ib_counter]) == -2) return (-1);	//EOL
//	if (ib_counter >= mCDCGetRxLength())               return (-1);	    //EOL
	while (trennzeichen(input_buffer[ib_counter]) ==  0)
	{
		rgetascii <<= 4;
		rgetascii += (input_buffer[ib_counter] - asciiKor);	
		ib_counter++;
	}
	return(0);
}	//getascii


//input_buffer -> dataPacket
//  #xx-xx-xx-xx-xx-xx
// einlesen der per RS232 empfangenen Zeichen
// beim Zeilenende wird befehl=1 gesetzt
// ergebnis sind maximal 16 bytes in dataPacket
// ab adsbScope 1.8 kommen nur maximal 4 Byte in einem 14 byte langen string '#XX-XX-XX-XX-<cr>'
void	AsciiToHexParser(void)
{
	for (ib_counter = 0; ib_counter < mCDCGetRxLength(); ib_counter++)
	{
		switch (trennzeichen(input_buffer[ib_counter]))
		{
			case -3:									//zeilenanfang #
				for (hb_counter = 0; hb_counter < 64; hb_counter++)
					dataPacket._byte[hb_counter] = 0;
				hb_counter  = 0;
				befehl      = 0;
				rgetascii   = 0;
				wertgueltig = 0;
				zeileangefangen = 1;
				break;
			case -2:									//zeilenende
				if (wertgueltig == 1)
				{
					dataPacket._byte[hb_counter] = rgetascii;
					hb_counter++;
					wertgueltig = 0;
				}
				rgetascii = 0;
				if (zeileangefangen == 1) befehl = 1;
				break;
			case -1:									//trennzeichen
				if (wertgueltig == 1)
				{
					dataPacket._byte[hb_counter] = rgetascii;
					hb_counter++;
					wertgueltig = 0;
				}
				rgetascii = 0;
				break;
			case 0: 									//ziffer
				rgetascii <<= 4;
				rgetascii += (input_buffer[ib_counter] - asciiKor);	
				wertgueltig = 1;						
				break;
		} //switch
	}
}	//AsciiToHexParser


// wandelt Zahl (0..15) in Zeichen ('0' .. 'A')
char	zeichen( byte wert)
{
	if (wert < 10) return (wert+'0');
	return (wert-10+'A');
}


//dataPacket -> input_buffer
// wird 52 bytes lang
void	HexToAsciiParser(void)
{
	char	puffer;
	for (ib_counter= 0; ib_counter<64; ib_counter++)
		input_buffer[ib_counter] = 0;
	ib_counter = 1;
	hb_counter = 0;
    // das @-zeichen ist fuer @TAG RAW; reserviert. 
	// ich sollte nebenwirkungsfrei auf #, & oder % wechsen koennen
	//input_buffer[0] = '@';
	input_buffer[0] = '#';
	for (hb_counter= 0; hb_counter<16; hb_counter++)
	{
		puffer = dataPacket._byte[hb_counter];
		input_buffer[ib_counter] = zeichen((puffer & 0xF0) >> 4);
		ib_counter++;
		input_buffer[ib_counter] = zeichen( puffer & 0x0F);
		ib_counter++;
		input_buffer[ib_counter] = '-';
		ib_counter++;
	}
	input_buffer[ib_counter] = 0x0A;
	ib_counter++;
	input_buffer[ib_counter] = 0x0D;
	ib_counter++;
	input_buffer[ib_counter] = 0x00;
	counter = ib_counter;
}	//HexToAsciiParser

// A S C I I * E N D E ***********************************************************************





void ServiceRequests(void)
{
	word	big_counter;		// fuer reset-warteschleife
	char	LookEnde;

    // K O M M A N D O  auswerten und ausfhren **************************************************

	if (UsbSelected)
	{
    	// wurde via USB etwas empfangen?
    	if(getsUSBUSART((char*)&input_buffer,sizeof(input_buffer)))  //byte getsUSBUSART(char *buffer, byte len)	//cdc ascii
			AsciiToHexParser();
	} else {
		if(getsRS232USART((char*)&input_buffer,sizeof(input_buffer))) 
			AsciiToHexParser();
	}

	if (befehl==1)
    {
        counter = 0;
		befehl  = 0;

		switch(dataPacket.CMD) 		//das ist in Assembler ein riesieger Sprungverteiler 
        {
			// eine Kennummer aus dem PIC auslesen
            case READ_VERSION:
                //dataPacket._byte[1] is len
                dataPacket._byte[2] = MINOR_VERSION; // Firmware-Version steht in user.h
                dataPacket._byte[3] = MAJOR_VERSION; // Hardware-Version
                counter=0x04;
                break;


			// nur DEBUG
			case SYS_KEY:
				adsb_CRC();
				dataPacket._byte[1] = CRC.byte2;
				dataPacket._byte[2] = CRC.byte1;
				dataPacket._byte[3] = CRC.byte0;
				counter = 4;	
				break;


			// betriebsart einstellen
			// 0 - OFF
			// 1 - reserved
			// 2 - all received data
			// 3 - only DF17
			// 4 - only DF17 + CRC-ok
			case SET_MODE:
				ADSB_Mode   =  dataPacket._byte[1] & 0x0F;
				TimeCode    = (dataPacket._byte[1] & 0x10);	// Zeitmarken senden
				SendFrameNr = (dataPacket._byte[1] & 0x20);	// framenumber senden
				SendHeartBeat=(dataPacket._byte[1] & 0x40);	// heartbeat senden
				counter = 2;
   				break;


			// AGC automatische Pegeleinstellung
			case SET_OFFSET:
				agc_offset    =  dataPacket._byte[3];			// offset in mV
				agc_mode      = (dataPacket._byte[2] & 0x01);	// 0=agc-off  1= agc-on   
				near_far_mode = (dataPacket._byte[2] & 0x06);	// 0=off  2,4,6= on       

				// offset im EEPROM auf 0x00 speichern
				EECON1bits.EEPGD = 0;				/* WRITE step #1 */
				EECON1bits.WREN  = 1;				/* WRITE step #2 */
				EEADR = 0x00;						/* WRITE step #3 */
				EEDATA = agc_offset;				/* WRITE step #4 */
				EECON2 = 0x55;						/* WRITE step #5 */
				EECON2 = 0xaa;						/* WRITE step #6 */
				EECON1bits.WR = 1;					/* WRITE step #7 */
				while (!PIR2bits.EEIF)				/* WRITE step #8 */
					;
				PIR2bits.EEIF = 0;					/* WRITE step #9 */

				// agc_mode invers im EEPROM auf 0x03 speichern
				EECON1bits.EEPGD = 0;				/* WRITE step #1 */
				EECON1bits.WREN  = 1;				/* WRITE step #2 */
				EEADR = 0x03;						/* WRITE step #3 */
				EEDATA = 1;							/* WRITE step #4 */
				if (agc_offset)	EEDATA = 0;
				EECON2 = 0x55;						/* WRITE step #5 */
				EECON2 = 0xaa;						/* WRITE step #6 */
				EECON1bits.WR = 1;					/* WRITE step #7 */
				while (!PIR2bits.EEIF)				/* WRITE step #8 */
					;
				PIR2bits.EEIF = 0;					/* WRITE step #9 */
				break;


			case SET_NEARDC:
				pwm_near_dc = (dataPacket._byte[1]);
				pwm_near_dc |=(dataPacket._byte[2] << 8);
				counter = 3;
				break;


			// AGC-offset auslesen
            case READ_OFFSET:
                dataPacket._byte[2] = agc_mode;  	// 1= agc-on   0=agc-off
                dataPacket._byte[3] = agc_offset; 	// offset in mV
                counter=0x04;
                break;


			// DC einstellen
			// ist sinnlos, falls AGC laeuft
			case SYS_PWM2:
				// [2] - lower 8 bit
				// [3] - upper 2 bit
				pwm_dutycycle =   dataPacket._byte[3];
				pwm_dutycycle <<= 8;
				pwm_dutycycle |=  dataPacket._byte[2];
				SetDCPWM2(pwm_dutycycle);  // 10 Bit
   				break;


			case SYS_ADC:
				switch (dataPacket._byte[1])
				{
					case 0:
						ADCON0 = 0x01;				// AN0, idle, enabled (Referenz)
						break;
					case 1:
						ADCON0 = 0x05;				// AN1, idle, enabled  (HF-Pegel)
						break;
				}
                counter=0x2;
				break;


			case RD_ADC:
				ADCON0bits.GO = 1;              // Start AD conversion
   				while(ADCON0bits.NOT_DONE);     // Wait for conversion
                dataPacket._byte[2] = ADRESL;
                dataPacket._byte[3] = ADRESH;
                counter=4;
				break;


			case SYS_EEPROM:
				// verwendet zum Entwerten der Firmware 
				UP_EEPROM();
   				break;


            case RESET:
                //When resetting, make sure to drop the device off the bus
                //for a period of time. Helps when the device is suspended.
                UCONbits.USBEN = 0;
                big_counter = 0;		//word = 16 Bit
                while(--big_counter);
				SleepMs(250);
				SleepMs(250);
				SleepMs(250);
				SleepMs(250);
                Reset();
                break;

                
            default:
				dataPacket._byte[0] = 0xFF;		// Error, unbekannter Befehl
				counter = 4;
                break;
        }	//end switch()


		// Antwort auf Befehl senden
		if(counter != 0)
		{
			HexToAsciiParser();	//wandlung in ASCII-String     dataPacket -> input_buffer    wird 52 bytes lang
			if (UsbSelected)
			{
				if(mUSBUSARTIsTxTrfReady()) mUSBUSARTTxRam((byte*)&input_buffer,counter);
			} else 
			{
				input_buffer[counter] = 0;	// 0-terminieren
				if(mRS232USARTIsTxTrfReady()) mUSBUSARTTxRam((byte*)&input_buffer,counter);
			}
		}
    }	//end if wurde via USB entwas empfangen 

    // K O M M A N D O  auswerten und ausfhren E N D E *********************************************




	// folgende Routinen werden bei jedem Schleifendurchlauf durchlaufen

    // Zeitmarke
	// die oberen 32 Bit der 48-Bit Zeitmarke aktualisieren
	TimeL.byte0 = TMR1L;			// triggert das lesen von TMR1H
	TimeL.byte1 = TMR1H;
	TimeL.byte2 = 0;
	TimeL._dword <<= 3;
	while (TimeL.byte2 != (TimeU.byte0 & 0x07)) TimeU._dword++;


	// alle 1,3 sec AGC
	// dauert ca 300 us
	// und HeartBeat senden
    if (INTCONbits.TMR0IF)
	{
		INTCONbits.TMR0IF = 0;

		// referenzpegel einstellen
//		if (agc_mode) pwm_avr();
		if (near_far_mode)
		{
			near_far_counter++;
			switch (near_far_counter)
			{
			case 1: 
				pwm_near();
				break;
			case 2: 
				pwm_far();
				break;
			case 3:
				pwm_avr(); 
				if (near_far_mode==2) near_far_counter = 0;
				break;
			case 4:
				pwm_avr(); 
				if (near_far_mode==4) near_far_counter = 0;
				break;
			case 5:
				pwm_avr(); 
				if (near_far_mode==6) near_far_counter = 0;
				break;
			default:	//failsave
				pwm_avr(); 
				near_far_counter = 0;
				break;
			}
		} else if (agc_mode) pwm_avr();

		//heartbeat senden
		if (SendHeartBeat)
		{
			adsb_heartbeat();
			adsb_StringOut(); 		// -> adsb_buffer
		}
	}


	// A D S B - F U N K T I O N E N *******************************************************

	// counter2 = 0;							// Anzahl der zu sendenden bytes

	//DF17 = RB3 = 08		0: DF17
	//CRC  = RB2 = 04		0: DF7 & CRC
	//AGC  = RB1 = 02		0: AGC ein
	//rem  = RB0 = 01		0: switches ignorieren

	// switch0 (remote) ist nicht gesteckt/geschlossen
	if (Switch0) 	// keine Fernsteuerung
	{
		ADSB_Mode=2;						// alles senden
		if (Switch_df17==0) ADSB_Mode=3;    // nur DF17 18 19
		if (Switch_crc ==0) ADSB_Mode=4;    // nur DF17 18 19 mit CRC
	}

	// @<12><14><14>;#<8>;<CR><LF><00>	ist 55 bytes lang
	// *<14><14>;#<8>;<CR><LF><00>		ist 43 bytes lang
	// @<12><14><14>;<CR><LF><00> 		ist 45 bytes lang
	// *<14><14>;<CR><LF><00> 			ist 33 bytes lang
	// *<14>;<CR><LF><00>				ist 19 bytes lang

	Limit = 0x7F - 19 - 14;					// Grundlaenge long frame 1+14+14+1+3

	if (!Switch_tag | TimeCode) Limit -= 12;	// 12 ZeichenZeitcode
	if (SendFrameNr)		    Limit -= 10;

	switch (ADSB_Mode)
	{
		case 1:		// USB-Bandbreitentest mit *...;  Format
			LATBbits.LATB7 = 1;			// LED2 an
			// 8D 3C 65 17 58 4D 10 5C FE 6B A1 E9 69 3A  - DF17:3C6517  Germany  Airb. Position 15225ft    *  Manchingen
			adsb[0]  = 0x8D;
			adsb[1]  = 0x3C;
			adsb[2]  = 0x65;
			adsb[3]  = 0x17;
			adsb[4]  = 0x58;
			adsb[5]  = 0x4D;
			adsb[6]  = 0x10;
			adsb[7]  = 0x5C;
			adsb[8]  = 0xFE;
			adsb[9]  = 0x6B;
			adsb[10] = 0xA1;
			adsb[11] = 0xE9;
			adsb[12] = 0x69;
			adsb[13] = 0x3A;
			do {
				adsb_StringOut(); 		// -> adsb_buffer
			} while (counter2<Limit);
			LATBbits.LATB7 = 0;			// LED2 aus
			break;

		case 2:		// *...;  Format
			adsbLoopH = 15;				// 15 = 30 ms
			adsbLoopL = 0;
			LATBbits.LATB7 = 1;			// LED2 an
			do {
				LookEnde = adsb_in();
				if (!LookEnde) adsb_StringOut(); 		// -> adsb_buffer
			} while (!LookEnde && (counter2<Limit));
			LATBbits.LATB7 = 0;			// LED2 aus
			break;

		case 3:		//DF=17		*...;  Format
			adsbLoopH = 15;				// 15 = 30 ms
			adsbLoopL = 0;
			LATBbits.LATB7 = 1;			// LED2 an
			do {
				LookEnde = adsb_in();
				if (!LookEnde) {
					DF = adsb_DF();
					if (DF == 17)      adsb_StringOut();	// -> adsb_buffer
					else if (DF == 18) adsb_StringOut();
					else if (DF == 19) adsb_StringOut();
				}
			} while (!LookEnde && (counter2<Limit));
			LATBbits.LATB7 = 0;			// LED2 aus
			break;

		case 4:		//DF=17		*...;  Format    CRC
			adsbLoopH = 15;				// 15 = 30 ms
			adsbLoopL = 0;
			LATBbits.LATB7 = 1;			// LED2 an
			do {
				LookEnde = adsb_in();
				if (!LookEnde) {
					DF = adsb_DF();

					if ((DF == 17) || (DF == 18) || (DF == 19))
					{
						if (adsb_CRC_OK()) adsb_StringOut();	// -> adsb_buffer
					}
				}
			} while (!LookEnde && (counter2<Limit));
			LATBbits.LATB7 = 0;			// LED2 aus
			break;
	}	//	switch (ADSB_Mode)

	// adsb daten via usb senden
	// mUSBUSARTTxRam dauert genau 1 Mikrosekunde
	if(counter2 != 0)		// sind bytes zu senden?
	{
		if (UsbSelected)
		{
			if (mUSBUSARTIsTxTrfReady())
			{
				mUSBUSARTTxRam((byte*)&adsb_buffer+c2Offset,counter2);
				counter2       = 0;
				if (c2Offset == 0) c2Offset = 0x80; else c2Offset = 0x00;
				LATCbits.LATC2 = 0;			// Frame_LED aus
			}
		} else
		{
			if(mRS232USARTIsTxTrfReady()) 
			{
				mRS232USARTTxRam((byte*)&adsb_buffer+c2Offset,counter2);
				counter2       = 0;
				if (c2Offset == 0) c2Offset = 0x80; else c2Offset = 0x00;
				LATCbits.LATC2 = 0;			// Frame_LED aus
			}
		}
	}

	// A D S B - F U N K T I O N E N   E N D E*************************************************

}//end ServiceRequests




/** E E P R O M **************************************************************************/


// 0x00 - Offsetspannung in mV
// 0x01 - RS232-Geschwindigkeit 0=115kb		1=1Mb			2=19.2kbit
// 0x02 - RS232-Polaritaet		0=driver	1=ohne driver
// 0x03 - PWM aktiv				0=aktiv		1=abgeschaltet
// ...
// 0xFF - Firmware gueltig?


// EEPROM, der interne EEPROM des PIC
// 0 - 
// 1 - 
// 2 - schreiben
// 3 - lesen
// 4 - Block schreiben
// 5 - Block lesen

// 0 - Klasse
// 1 - Befehl 2,3
// 2 - Adresse
// 3 - Daten

// 0 - Klasse
// 1 - Befehl 4,5
// 2 - Adresse
// 3 - Laenge
// 4..7 - Daten
void UP_EEPROM(void)
{
	byte	Nummer;
	word	Adresse;
	word	Ende;
	switch (dataPacket._byte[1])
	{
		case 2:
			EECON1bits.EEPGD = 0;				/* WRITE step #1 */
			EECON1bits.CFGS  = 0;
			EECON1bits.WREN = 1;				/* WRITE step #2 */
			EEADR  = dataPacket._byte[2];		/* WRITE step #3 */
			EEDATA = dataPacket._byte[3];		/* WRITE step #4 */
			EECON2 = 0x55;						/* WRITE step #5 */
			EECON2 = 0xaa;						/* WRITE step #6 */
			EECON1bits.WR = 1;					/* WRITE step #7 */
			while (!PIR2bits.EEIF);				/* WRITE step #8 */
			PIR2bits.EEIF = 0;					/* WRITE step #9 */
			break;
		case 3:
			EECON1bits.EEPGD = 0;				/* READ step #1 */
			EECON1bits.CFGS  = 0;
			EEADR = dataPacket._byte[2];		/* READ step #2 */
			EECON1bits.RD = 1;					/* READ step #3 */
			dataPacket._byte[3] = EEDATA;		/* READ step #4 */
			break;
		case 4:
			Nummer = 4;
			Ende   = dataPacket._byte[2];
			Ende  += dataPacket._byte[3];
			for (Adresse=dataPacket._byte[2]; Adresse<Ende; Adresse++)
			{
				EECON1bits.EEPGD = 0;				/* WRITE step #1 */
				EECON1bits.CFGS  = 0;
				EECON1bits.WREN = 1;				/* WRITE step #2 */
				EEADR = Adresse;					/* WRITE step #3 */
				EEDATA = dataPacket._byte[Nummer];	/* WRITE step #4 */
				EECON2 = 0x55;						/* WRITE step #5 */
				EECON2 = 0xaa;						/* WRITE step #6 */
				EECON1bits.WR = 1;					/* WRITE step #7 */
				while (!PIR2bits.EEIF);				/* WRITE step #8 */
				PIR2bits.EEIF = 0;					/* WRITE step #9 */
				Nummer+=1;
			}
			break;
		case 5:
			Nummer = 4;
			Ende   = dataPacket._byte[2];
			Ende  += dataPacket._byte[3];
			for (Adresse=dataPacket._byte[2]; Adresse<Ende; Adresse++)
			{
				EECON1bits.EEPGD = 0;				/* READ step #1 */
				EECON1bits.CFGS  = 0;
				EEADR = Adresse;					/* READ step #2 */
				EECON1bits.RD = 1;					/* READ step #3 */
				dataPacket._byte[Nummer] = EEDATA;	/* READ step #4 */
				Nummer += 1;
			}
			break;
	}
	counter = 16;
} // UP_EEPROM





/****  W A R T E S C H L E I F E N **************************************************************/

// verzgerungen bis max. 63 ms
// Resonator 20 MHz, CPU-Takt 48 MHz, Cycl. 12 MHz
// 1 us = 12 Zyklen
// 1 ms = 12*1000 Zyklen
// High-Teil x 3072
// Low Teil x 12
// die delay-Routinen vertragen als input nur  1..255
void SleepUs(unsigned int us)
{
	byte teilzeit;
	teilzeit = us & 0x00FF;
    if (teilzeit>0)
	{
		Delay10TCYx(teilzeit);		// 10 Zyklen ~ 1s
	}
	teilzeit = us >> 8;			// 1/256
    if (teilzeit>0)
	{
		Delay1KTCYx(teilzeit);	// 3070 Zyklen ~258s ~0,25ms
		Delay1KTCYx(teilzeit);
		Delay1KTCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
		Delay10TCYx(teilzeit);
	}
} //SleepUs



// verzgerungen bis max. 255 ms
// Resonator 20 MHz, CPU-Takt 48 MHz, Cycl. 12 MHz
// 1 us = 12 Zyklen
// 1 ms = 12*1000 Zyklen
// 4 ms = 48x1000 Zyklen
// die delay-Routinen vertragen als input nur  1..255
void SleepMs(unsigned int ms)
{
    if (ms>0)
	{
		Delay10KTCYx(ms);
		Delay1KTCYx(ms);
		Delay1KTCYx(ms);
	}
} //SleepMs



/****  E E P R O M   F U N K T I O N E N  F U E R   S T E U E R P I C ********************/


//lesen von Daten aus dem EEPROM des Steuerpic
//0 - Befehl
//1 - Startadresse -low
//2 - Startadresse high = 0
//3 - Blocklnge
void readEdata(void)
{
	byte	Nummer;
	word	Adresse;
	dataPacket._byte[0] = READ_EDATA;
	counter = 63;
	Nummer  = 4;
	for (Adresse=dataPacket._byte[1]; Adresse<dataPacket._byte[1]+dataPacket._byte[3]; Adresse++)
	{
		EECON1bits.EEPGD = 0;				/* READ step #1 */
		EEADR = Adresse;					/* READ step #2 */
		EECON1bits.RD    = 1;				/* READ step #3 */
		dataPacket._byte[Nummer] = EEDATA;	/* READ step #4 */
		Nummer += 1;
	}
}


//lesen von Daten aus dem EEPROM des Steuerpic
//0 - Befehl
//1 - Startadresse -low
//2 - Startadresse high = 0
//3 - Blocklnge
//4 - 1. Datenbyte
void writeEdata(void)
{
	byte	Nummer;
	word	Adresse;
	counter = 0x01;
	dataPacket._byte[0] = WRITE_EDATA;
	Nummer = 4;
	for (Adresse=dataPacket._byte[1]; Adresse<dataPacket._byte[1]+dataPacket._byte[3]; Adresse++)
	{
		EECON1bits.EEPGD = 0;				/* WRITE step #1 */
		EECON1bits.WREN  = 1;				/* WRITE step #2 */
		EEADR = Adresse;					/* WRITE step #3 */
		EEDATA = dataPacket._byte[Nummer];	/* WRITE step #4 */
		EECON2 = 0x55;						/* WRITE step #5 */
		EECON2 = 0xaa;						/* WRITE step #6 */
		EECON1bits.WR = 1;					/* WRITE step #7 */
		while (!PIR2bits.EEIF)				/* WRITE step #8 */
			;
		PIR2bits.EEIF = 0;					/* WRITE step #9 */
		Nummer+=1;
	}
}



/****  A D C   F U N K T I O N E N  ********************************************************/

// Spannung an AN1 auslesen *************************
void ReadADC(void)
{
	ADCON0bits.GO = 1;              // Start AD conversion
   	while(ADCON0bits.NOT_DONE);     // Wait for conversion
    return;
}//end ReadADC






/** EOF user.c ***************************************************************/
