/*********************************************************************
 * FileName:        rs232.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Company:         sprut
 * Copyright:       2007-2010 Joerg Bredendiek (sprut)
 *
 *
 ********************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include <usart.h>
#include "system\typedefs.h"
#include "system\usb\usb.h"
#include "system\usb\class\cdc\cdc.h"
#include "io_cfg.h"             	// I/O pin mapping

#include "delays.h"
//#include "system\interrupt\interrupt.h"
#include "user\user.h"
//#include "user\adsbin.h"
#include "user\rs232.h"

//#include "timers.h"



/** V A R I A B L E S ********************************************************/
#pragma udata



/** D E C L A R A T I O N S **************************************************/
#pragma code


// 115.2 kbaud = 14,4 kByte/ s  69 us/Byte = 2 ms/DF17 = 480 fps = 28800 fpm 
// receiver sollte int ausloesen knnen, oder ich kann nur 1-byte-kommandos senden
// speed=0 : 115.200
// speed=1 : 921.600
// speed=2 :  19.200
//
// BRGH=1
// Calculated Baud Rate = 48000000/(16 (25 + 1)) = 115385
// Calculated Baud Rate = 3.000.000/(25 + 1) = 115385 = 101,6% of 115.2kbaud

// BRG16=BRGH=1
// Calculated Baud Rate = 48000000/(4 (xx + 1)) = 923077
// Calculated Baud Rate = 12.000.000/(12 + 1) = 923077 = 100.16% of 921600

// BRGH=1
// Calculated Baud Rate = 48000000/(16 (155 + 1)) = 19231
// Calculated Baud Rate = 3.000.000/(155 + 1) = 19231 = 101,6% of 19.2kbaud


void rs232_init(void)
{
	byte	speed;
	byte	polaritaet;

	// switch0 (rem) ist nicht gesteckt/geschlossen
	if (Switch0) 	// keine Fernsteuerung
	{
		speed = 0;	//default 115 kBaud
		if (!Switch_1mbit) speed = 1;	//1 MBit
	}
	else
	{
		// RS232-speed aus dem EEPROM laden; 0 = 115 kBaud;  1 = 1 MBit
    	EECON1 = 0x00;
		EEADR  = 0x01;			//Adresse = 0x01
		EECON1_RD = 1;
		speed = EEDATA;
		if (speed>2) speed=0;
	}

	if (speed==0)
	{
		// BRGH = 1   SPBRG=25  -> 115384 baud = 101,6% of 115.2kbaud
		OpenUSART(	USART_TX_INT_OFF &
					USART_RX_INT_ON  &
					USART_ASYNCH_MODE &
					USART_EIGHT_BIT &
					USART_CONT_RX &		// USART_SINGLE_RX
					USART_BRGH_HIGH,
					25 );
		BAUDCONbits.BRG16 = 0;
	}
	if (speed==1)
	{
		// BRG16=BRGH=1   SPBRG=12  -> 923077 baud 
		OpenUSART(	USART_TX_INT_OFF &
					USART_RX_INT_ON  &
					USART_ASYNCH_MODE &
					USART_EIGHT_BIT &
					USART_CONT_RX &		// USART_SINGLE_RX
					USART_BRGH_HIGH,
					12 );
		BAUDCONbits.BRG16 = 1;
	}
	if (speed==2)
	{
		// BRGH = 1   SPBRG=155  -> 19231 baud
		OpenUSART(	USART_TX_INT_OFF &
					USART_RX_INT_ON  &
					USART_ASYNCH_MODE &
					USART_EIGHT_BIT &
					USART_CONT_RX &		// USART_SINGLE_RX
					USART_BRGH_HIGH,
					155 );
		BAUDCONbits.BRG16 = 0;
	}


	// RS232-polaritaet aus dem EEPROM laden; 
	// 0 = data is not inverted (brauche Treiber);
	// 1 = data is inverted (geht ohne Treiber
    EECON1 = 0x00;
	EEADR  = 0x02;			//Adresse = 0x02
	EECON1_RD = 1;
	polaritaet = EEDATA;
	if (polaritaet>1) polaritaet=0;

	if (polaritaet==0)
	{
	    // das ist der Normalfall mit externen Treibern
		BAUDCON = BAUDCON & 0xCF;
	} else{
	    // ohne Treiber
		BAUDCON = BAUDCON | 0x30;
	}


    cdc_trf_state = CDC_TX_READY;
    cdc_rx_len = 0;

	PIR1bits.RCIF 	= 0;
	PIE1bits.RCIE	= 1;
	INTCONbits.PEIE = 1;
	INTCONbits.GIE  = 1;
}



void RS232_tx(void)
{
	//WriteUSART( 00 );
	//putsUSART( char *data );  //sendet 0terminierten string


	cdc_trf_state = CDC_TX_READY;
}



void RS232_rx(void)
{
	// empfangene daten nach input_buffer schreiben (maximal 64 byte)
	// Anzahl der empfangenen bytes nach cdc_rx_len schreiben


}


void RS232Tasks(void)
{
	RS232CDCTxService();
	//RS232_tx();
	//RS232_rx();
}

/******************************************************************************
 * Function:        byte getsRS232USART(char *buffer,
 *                                    byte len)
 *
 * PreCondition:    Value of input argument 'len' should be smaller than the
 *                  maximum endpoint size responsible for receiving bulk
 *                  data from USB host for CDC class.
 *                  Input argument 'buffer' should point to a buffer area that
 *                  is bigger or equal to the size specified by 'len'.
 *
 * Input:           buffer  : Pointer to where received bytes are to be stored
 *                  len     : The number of bytes expected.
 *
 * Output:          The number of bytes copied to buffer.
 *
 * Side Effects:    Publicly accessible variable cdc_rx_len is updated with
 *                  the number of bytes copied to buffer.
 *                  Once getsUSBUSART is called, subsequent retrieval of
 *                  cdc_rx_len can be done by calling macro mCDCGetRxLength().
 *
 * Overview:        getsUSBUSART copies a string of bytes received through
 *                  USB CDC Bulk OUT endpoint to a user's specified location. 
 *                  It is a non-blocking function. It does not wait
 *                  for data if there is no data available. Instead it returns
 *                  '0' to notify the caller that there is no data available.
 *
 * Note:            If the actual number of bytes received is larger than the
 *                  number of bytes expected (len), only the expected number
 *                  of bytes specified will be copied to buffer.
 *                  If the actual number of bytes received is smaller than the
 *                  number of bytes expected (len), only the actual number
 *                  of bytes received will be copied to buffer.
 *****************************************************************************/
byte getsRS232USART(char *buffer, byte len)
{
    cdc_rx_len = 0;

	if (RS232_rx_EOL)
	{
        // Adjust the expected number of bytes to equal the actual number of bytes received.
        if(len > RS232_rx_count)
            len = RS232_rx_count;

        // Copy data from dual-ram buffer to user's buffer
        for(cdc_rx_len = 0; cdc_rx_len < len; cdc_rx_len++)
            buffer[cdc_rx_len] = cdc_data_rx[cdc_rx_len];

		RS232_rx_count	= 0;
		RS232_rx_EOL	= 0;
	}
    
    return cdc_rx_len;
    
}//end getsRS232USART

/******************************************************************************
 * Function:        void RS232CDCTxService(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        RS232CDCTxService handles device-to-host transaction(s).
 *                  This function should be called once per Main Program loop.
 *
 * Note:            None
 *****************************************************************************/
void RS232CDCTxService(void)
{
	if 	(cdc_trf_state == CDC_TX_READY) return;

	putsUSART( pCDCSrc.bRam );  		//sendet 0-terminierten string
	cdc_trf_state = CDC_TX_READY;


}//end RS232CDCTxService


/** EOF rs232.c ***************************************************************/
